/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Object.h
 *    $Defines object class as base class for classFactory compatible classes$.
 *
 * @author Ronny Stricker
 * @date   2010/10/20
 */

#ifndef _MIRA_OBJECT_H_
#define _MIRA_OBJECT_H_

#include <string>

///@cond INTERNAL
// class which will be abused as compile time error message
template< class T >
class ________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR1__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________
{
	virtual int FOR_CLASS();
	virtual ~________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR1__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________() {}
};

// class which will be abused as compile time error message
template< class T >
class ________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR2__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________
{
	virtual int FOR_CLASS();
	virtual ~________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR2__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________() {}
};

// class which will be abused as compile time error message
template< class T >
class ________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR3__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________
{
	virtual int FOR_CLASS();
	virtual ~________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR3__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________() {}
};

// class which will be abused as compile time error message
template< class T >
class ________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR4__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________
{
	virtual int FOR_CLASS();
	virtual ~________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR4__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________() {}
};

///@endcond

namespace mira {

//////////////////////////////////////////////////////////////////////////////

class Class;
class Object;

// *****************************************************************************
// *** PseudoClass
// *****************************************************************************

/**
 * @brief This is an auxiliary to enable more verbose exception messages.
 * Since the exception messages simply tries to call the getIdentifier()
 * function, the Object class is required to have one.
 */
class PseudoClass
{
public:
	std::string const& getIdentifier() const
	{
		static std::string tIdent("mira::Object");
		return tIdent;
	}

	/**
	 * @brief Return a new instance of the class with the given identifier.
	 * @throw XFactoryLogical If the class is abstract.
	 * @throw XFactoryLogical If the class has no default constructor
	 */
	Object* newInstance( std::string const& childIdentifier ) const;

	/**
	 * @brief Return a new instance of the class with the given identifier.
	 * The class is casted to the desired class type.
	 * @throw XFactoryBadCast If the cast fails
	 * @throw XFactoryLogical If the class is abstract.
	 * @throw XFactoryLogical If the class has no default constructor
	 */
	template <class CLASS>
	CLASS* newInstance( std::string const& childIdentifier ) const;

};

// *****************************************************************************
// *** Object
// *****************************************************************************

/**
 * @brief The object class acts as a generic base class for classes which should
 * be used with the classFactory. To add a minimum of extra functions to the
 * derived classes, the object class only implements functions to retrieve the
 * associated class object. All functionalities are encapsulated in the class
 * object.
 *
 * @ingroup ClassFactoryModule
 */
class Object
{
	friend class ClassFactory;

	template<typename Class, typename Base>
	friend struct FactoryRegisterClassHelper;

public:
	virtual ~Object()
	{
	}

	/**
	 * @brief call the virtual internalGetClass().
	 */
	Class const& getClass() const {
		return internalGetClass();
	}

	static PseudoClass const& CLASS() {
		static mira::PseudoClass sClass;
		return sClass;
	}

protected:
	virtual Class const& internalGetClass() const = 0;

};

// *****************************************************************************
// *** MACROS
// *****************************************************************************

// the macros should be used to enable the class factory to use the special
// constructors

#define MIRA_OBJECT_CONSTRUCTOR1(T, A0) \
template <> int \
	________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR1__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________ \
	<T>::FOR_CLASS() { return 0; } \
namespace mira { \
	template<> \
	T* mira::ClassFactoryDefaultConstClassBuilder::invoke<1,T>( std::va_list ap) { \
		A0 a0 = va_arg(ap, A0); \
		return new T(a0); \
	} \
}

#define MIRA_OBJECT_CONSTRUCTOR2(T, A0, A1) \
template <> int \
	________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR2__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________ \
	<T>::FOR_CLASS() { return 0; } \
namespace mira { \
	template<> \
	T* mira::ClassFactoryDefaultConstClassBuilder::invoke<2,T>( std::va_list ap) { \
		A0 a0 = va_arg(ap, A0); \
		A1 a1 = va_arg(ap, A1); \
		return new T( a0, a1 ); \
	} \
}

#define MIRA_OBJECT_CONSTRUCTOR3(T, A0, A1, A2) \
template <> int \
	________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR3__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________ \
	<T>::FOR_CLASS() { return 0; } \
namespace mira { \
	template<> \
	T* mira::ClassFactoryDefaultConstClassBuilder::invoke<3,T>( std::va_list ap) { \
		A0 a0 = va_arg(ap, A0); \
		A1 a1 = va_arg(ap, A1); \
		A2 a2 = va_arg(ap, A2); \
		return new T( a0, a1, a2 ); \
	} \
}

#define MIRA_OBJECT_CONSTRUCTOR4(T, A0, A1, A2, A3) \
template <> int \
	________________________________PLEASE_USE_THE__MIRA_OBJECT_CONSTRUCTOR4__MACRO_IN_GLOBAL_NAMESPACE_ONLY________________________________ \
	<T>::FOR_CLASS() { return 0; } \
namespace mira { \
	template<> \
	T* mira::ClassFactoryDefaultConstClassBuilder::invoke<4,T>( std::va_list ap) { \
		A0 a0 = va_arg(ap, A0); \
		A1 a1 = va_arg(ap, A1); \
		A2 a2 = va_arg(ap, A2); \
		A3 a3 = va_arg(ap, A3); \
		return new T( a0, a1, a2, a3 ); \
	} \
}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_OBJECT_H_ */

