/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LoggingAux.h
 *    Auxiliary logging macros for special entities like exceptions, etc.
 *
 * @author Erik Einhorn
 * @date   2011/02/27
 */

#ifndef _MIRA_LOGGINGAUX_H_
#define _MIRA_LOGGINGAUX_H_

#include <exception>

#include <error/SignalHandler.h>
#include <error/LoggingCore.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace Private {

void logAuxLogger(const std::exception& info,
                  LogRecord& record, std::ostringstream& os);
void logAuxLogger(const IntSignal& info,
                  LogRecord& record, std::ostringstream& os);

template <typename T>
class AuxLogger : boost::noncopyable
{
public:
	AuxLogger(SeverityLevel level, T auxInfo) : mAuxInfo(auxInfo) {
		mRecord.level = level;
		mRecord.time = Time::now();
		mRecord.uptime = MIRA_LOGGER.getUptime();
	}

	~AuxLogger() {
		logAuxLogger(mAuxInfo, mRecord, mStream);
	}

	/// Get underlying stream
	std::ostringstream& stream() {
		return mStream;
	}

protected:

	LogRecord mRecord;
	std::ostringstream mStream;
	T mAuxInfo;
};

template class MIRA_BASE_EXPORT AuxLogger<const std::exception&>;
template class MIRA_BASE_EXPORT AuxLogger<const IntSignal&>;
}
///@endcond

//////////////////////////////////////////////////////////////////////////////

/**
 * Log the specified exception, including all information that the exception
 * object carries.
 * The usage is similar to MIRA_LOG.
 *
 * @ingroup LoggingModule
 */
#define MIRA_LOG_EXCEPTION(level, ex)                                         \
if (level > MIRA_SEVERITY_MAX_LEVEL) {}                                       \
else if (level > MIRA_LOGGER.getSeverityLevel()) {}                           \
	else mira::Private::AuxLogger<const std::exception&>(level, ex).stream()



/**
 * Log the error string based on the specified signal and a backtrace
 * if available.
 * The usage is similar to MIRA_LOG.
 *
 * @ingroup LoggingModule
 */
#define MIRA_LOG_SIGNAL(level, sig)                                           \
if (level > MIRA_SEVERITY_MAX_LEVEL) {}                                       \
else if (level > MIRA_LOGGER.getSeverityLevel()) {}                           \
	else mira::Private::AuxLogger<const IntSignal&>(level, sig).stream()

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_LOGGINGAUX_H_ */
