/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CallStack.h
 *    Encapsulates call stack functionality.
 *
 * @author Erik Einhorn, Christian Martin
 * @date   2009/12/26
 */

#ifndef _MIRA_CALLSTACK_H_
#define _MIRA_CALLSTACK_H_

#include <platform/Types.h>
#include <serialization/GetterSetter.h>

#ifdef MIRA_LINUX
# include <execinfo.h>
#endif

#ifdef MIRA_WINDOWS
# include <DbgHelp.h>
#endif

#include <stdlib.h>
#include <stdio.h>

#include <iostream>
#include <string>
#include <vector>

#ifndef Q_MOC_RUN
#include <boost/shared_array.hpp>
#include <boost/lexical_cast.hpp>
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Encapsulates unix call stack functionality.
 * It allows to retrieve the current call stack with all symbols to
 * the functions.
 * The stack can be printed to a stream in a similar format as
 * in the GDB using the << operator.
 *
 * @ingroup ExceptionModule
 */
class MIRA_BASE_EXPORT CallStack
{
protected:
#ifdef MIRA_LINUX
	/// Constructor for internal use only
	CallStack(void** stack, std::size_t stackSize, std::size_t stackStart) :
		mStack(stack), mStackSize(stackSize),
		mStackStart(stackStart) {}
#endif
#ifdef MIRA_WINDOWS
	/// Constructor for internal use only
	CallStack(const std::vector<STACKFRAME64>& stack, 
	          std::size_t stackSize, std::size_t stackStart) :
		mStack(stack), mStackSize(stackSize),
		mStackStart(stackStart) {}
#endif

public:

	/// Constructor that creates an empty call stack.
	CallStack() : mStackSize(0), mStackStart(0) {}

	/// Copy-Constructor
	CallStack(const CallStack& other) :
		mStack(other.mStack), mStackSize(other.mStackSize),
		mStackStart(other.mStackStart), mSymbols(other.mSymbols) {}

	/// Assignment operator with similar behaviour as copy-constructor.
	const CallStack& operator=(const CallStack& other);

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		// mStack is only required to obtain the actual symbols,
		// no need to reflect it (luckily, as shared_array is not supported yet)
		r.property("stacksize", mStackSize, "");
		r.property("stackstart", mStackStart, "");
		r.property("symbols",
		           getter(&CallStack::getSymbols, this),
		           setter(&CallStack::setSymbols, this),
		           "");
	}

public:

	/**
	 * Contains all information of a single function symbol in the
	 * call stack.
	 */
	struct Symbol
	{
		Symbol() :
			address(0),
			offset(0) {}

		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("address", address, "");
			r.property("addressStr", addressStr, "");
			r.property("offset", offset, "");
			r.property("name", name, "");
			r.property("nameDemangled", nameDemangled, "");
			r.property("file", file, "");
		}

		/// The address of the code portion as a 64 bit integer
		int64 address;
		/// The address of the code portion in a "human readable" format.
		std::string addressStr;
		/// The corresponding offset.
		int64 offset;
		/// The mangled name of the function
		std::string name;
		/// The demangled name of the function (on Linux only available, if 
		/// CALLSTACK_LINUX_USE_DEMANGLE is defined, otherwise equal to name)
		std::string nameDemangled;
		/// the binary file
		std::string file;

		/**
		 * Tries to obtain the source file name and line number for the given 
		 * symbol using the debug information and the symbol table.
		 * On Linux: 
		 *    Only available if CALLSTACK_LINUX_USE_SYMBOLTABLE is defined and
		 *    debug information is compiled into the binaries.
		 * On Windows: 
		 *    Only available if debug information is compiled into the binaries.
		 */
		std::string sourceLocation() const;

		/**
		 * Prints the symbol in the similar format as the GDB shows function symbols
		 * into the stream.
		 */
		friend MIRA_BASE_EXPORT std::ostream& operator<<(std::ostream& os, const Symbol& s);
	};

	/// Returns the i-th symbol of the call stack
	const Symbol& operator[](std::size_t i) const;

	/// Returns the size of the call stack
	std::size_t size() const { return mStackSize-mStackStart; }

	/// Prints the whole stack into a stream in a similar format as the GDB
	friend MIRA_BASE_EXPORT std::ostream& operator<<(std::ostream& os, const CallStack& stack);

public:
	/**
	 * Creates the call stack object that contains the callstack
	 * starting at the current address of the instruction pointer.
	 */
	static CallStack backtrace(std::size_t maxSize = 10, std::size_t stackStart = 1);

private:
	/// parses the symbols and extracts information
	void obtainSymbols() const;

	const std::vector<Symbol>& getSymbols();
	void setSymbols(const std::vector<Symbol>& symbols);

private:
#ifdef MIRA_LINUX
	boost::shared_array<void*> mStack;
#endif
#ifdef MIRA_WINDOWS
	std::vector<STACKFRAME64> mStack;
#endif
	std::size_t mStackSize;
	std::size_t mStackStart;

	mutable std::vector<Symbol> mSymbols; // the symbols of the call stack
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
