/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeDataView.C
 *    Implementation of TapeDataView.h.
 *
 * @author Tim Langner
 * @date   2012/01/01
 */

#include <TapeDataView.h>

#include <QEvent>
#include <QApplication>
#include <QHelpEvent>
#include <QMenu>
#include <QRubberBand>
#include <QStyle>
#include <QToolButton>
#include <QToolTip>

#include <widgets/QtUtils.h>

#include <TapeDataRenderer.h>
#include <TapeEditor.h>
#include <TapeCommand.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TapeDataView::TapeDataView(QWidget* parent, TapeEditor* editor) :
	ScrollView(parent)
{
	mEditor = editor;
	mZoomIn    = new QToolButton(this);
	mZoomIn->setIcon(QIcon(":/icons/Plus.png"));
	mZoomOut   = new QToolButton(this);
	mZoomOut->setIcon(QIcon(":/icons/Minus.png"));
	mZoomReset = new QToolButton(this);
	mZoomReset->setIcon(QIcon(":/icons/Loop.png"));

	mZoomIn->setAutoRepeat(true);
	mZoomOut->setAutoRepeat(true);

	mZoomIn->setToolTip(tr("Zoom in (horizontal)"));
	mZoomOut->setToolTip(tr("Zoom out (horizontal)"));
	mZoomReset->setToolTip(tr("Zoom reset"));

	connect(mZoomIn, SIGNAL(clicked()), mEditor, SLOT(zoomIn()));
	connect(mZoomOut, SIGNAL(clicked()), mEditor, SLOT(zoomOut()));
	connect(mZoomReset, SIGNAL(clicked()), mEditor, SLOT(zoomReset()));

	int scrollBarExtent = style()->pixelMetric(QStyle::PM_ScrollBarExtent);
	mZoomIn->setFixedWidth(scrollBarExtent);
	mZoomOut->setFixedWidth(scrollBarExtent);
	addScrollBarWidget(mZoomIn,  Qt::AlignRight);
	addScrollBarWidget(mZoomOut, Qt::AlignRight);

	setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
	setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);

	viewport()->setFocusPolicy(Qt::StrongFocus);
	viewport()->setAcceptDrops(true);
	setMouseTracking(true);
	const QFont& font = ScrollView::font();
	setFont(QFont(font.family(), font.pointSize() - 1));
	viewport()->installEventFilter(this);

	mRubberBand = new QRubberBand(QRubberBand::Rectangle, viewport());

	mDragState = DragNone;
}

void TapeDataView::resizeEvent(QResizeEvent *resizeEvent)
{
	ScrollView::resizeEvent(resizeEvent);
	if (mZoomReset != NULL)
	{
		int h = size().height();
		int w = ScrollView::style()->pixelMetric(
			QStyle::PM_ScrollBarExtent);
		int x = size().width() - w - 2;
		mZoomReset->setGeometry(x, h - w - 2, w, w);
	}
	updateContents();

	mEditor->getThumbView()->updateThumb();
}

void TapeDataView::contextMenuEvent(QContextMenuEvent* contextMenuEvent)
{
	QPoint pos = viewportToContents(contextMenuEvent->pos());
	TapeChannelInfo* info = getChannelAtPos(pos);
	if (info != NULL)
	{
		QMenu menu;
		int64 time = mEditor->getTimeFromPixel(pos.x());
		TapeChannelInfo::DataMap::iterator item = getMessageAtPos(pos, *info);
		info->renderer->fillContextMenu(&menu, *info, item, time);

		QAction* deleteAction = NULL;
		if (item != info->data.end())
			deleteAction = menu.addAction(tr("Delete message"));
		QAction* action = menu.exec(contextMenuEvent->globalPos());
		if (action == NULL)
			return;
		if (deleteAction != NULL && action == deleteAction)
			mEditor->executeCommand(new EraseMessageCommand(info->name, item->first, mEditor));
		else
			info->renderer->executeAction(action, *info, item, time);
	}
}

void TapeDataView::mousePressEvent(QMouseEvent* mouseEvent)
{
	mDragState = DragNone;

	if (mouseEvent->button() == Qt::LeftButton)
	{
		QPoint pos = viewportToContents(mouseEvent->pos());
		TapeChannelInfo* info = getChannelAtPos(pos);
		mEditor->clearSelectedChannels();
		if (info != NULL)
			info->selected = true;
		mDragPos = pos;
		mDragState = DragStart;
		mEditor->setSelectionStart(mEditor->getTimeFromPixel(mDragPos.x()));
		mEditor->setSelectionEnd(mEditor->getTimeFromPixel(mDragPos.x()));
		mEditor->getChannelView()->updateContents();
	}
	ScrollView::mousePressEvent(mouseEvent);
}

void TapeDataView::mouseMoveEvent(QMouseEvent* mouseEvent)
{
	QPoint pos = viewportToContents(mouseEvent->pos());

	if (mDragState == DragStart &&
		(pos - mDragPos).manhattanLength() > QApplication::startDragDistance())
	{
		mDragState = DragMove;
		setCursor(QCursor(Qt::SizeHorCursor));
		QRect selection;
		selection.setTopLeft(mDragPos);
		selection.setBottomRight(pos);
		setSelection(selection);
		mRubberBand->show();
	}
	if (mDragState == DragMove)
	{
		ScrollView::ensureVisible(pos.x(), pos.y(), 24, 24);
		QRect selection;
		selection.setTopLeft(mDragPos);
		selection.setBottomRight(pos);
		setSelection(selection);

		int64 start = mEditor->getTimeFromPixel(mSelection.left());
		int64 end = mEditor->getTimeFromPixel(mSelection.right());
		QString txt("Start:\t");
		txt += mEditor->getStringFromTime(start) + "\nEnd:\t";
		txt += mEditor->getStringFromTime(end) + "\nLength:\t";
		txt += mEditor->getStringFromTime(end-start);
		QToolTip::showText(QCursor::pos(), txt, viewport());
	}
}

void TapeDataView::mouseReleaseEvent(QMouseEvent* mouseEvent)
{
	ScrollView::mouseReleaseEvent(mouseEvent);

	if (mDragState == DragStart || mDragState == DragMove)
	{
		mDragState = DragNone;
		mRubberBand->hide();
		mEditor->getTimeView()->viewport()->update();
		mEditor->getThumbView()->update();
		viewport()->update();
		unsetCursor();
	}
}

void TapeDataView::wheelEvent(QWheelEvent* event)
{

	// delegate the wheelEvent to the base class only, if the user
	// pressed Shift or Ctrl to scroll the view  ...
	if (event->modifiers() & (Qt::ShiftModifier | Qt::ControlModifier)) {
		ScrollView::wheelEvent(event);
		return;
	}

	/// ... otherwise scaling is the desired behaviour

	int numDegrees = event->delta() / 8;
	int numSteps = numDegrees / 15;  // see QWheelEvent documentation

	QPoint pos = viewportToContents(event->pos());
	int64 time = mEditor->getTimeFromPixel(pos.x());

	if (numSteps > 0)
		mEditor->zoomIn(numSteps);
	else
		mEditor->zoomOut(-numSteps);

	// now move the content to the position, so that the time that was
	// under the mouse curser before scaling, is again under the cursor
	setContentsPos(0, contentsY());
	pos = viewportToContents(event->pos());
	int left = std::max(mEditor->getPixelFromTime(time)-pos.x(), 0);
	setContentsPos(left, contentsY());
}


void TapeDataView::keyPressEvent(QKeyEvent* keyEvent)
{
	switch(keyEvent->key())
	{
		case Qt::Key_Delete : // delete messages in the selected interval
		{
			mEditor->eraseSelectedMessages();
			break;
		}
		default:
			ScrollView::keyPressEvent(keyEvent);
			break;
	}
}

void TapeDataView::updatePixmap(int cx, int cy)
{
	QWidget *viewport = this->viewport();
	int w = viewport->width();
	int h = viewport->height();

	const QPalette& pal = palette();

	if (w < 1 || h < 1)
		return;

	mPixmap = QPixmap(w, h);
	mPixmap.fill(pal.dark().color());

	QColor rgbLight = pal.midlight().color();
	QColor rgbMid   = pal.mid().color();
	QColor rgbDark  = rgbMid.darker(120);
	QColor timeIndicator = rgbLight;
	timeIndicator.setAlpha(160);

	QPainter painter(&mPixmap);
	QtUtils::initPainterFrom(painter, *this);

	TapeChannelInfoMap& channels = mEditor->getChannels();
	int x1, x2, lastX, y1, y2, h1, h2, w1;
	y1 = y2 = 0;
	foreach(TapeChannelInfoMap::value_type& channel, channels)
	{
		TapeDataRendererPtr renderer = channel.second.renderer;
		h1 = renderer->getTotalHeight();
		h2 = renderer->getDataHeight();
		w1 = renderer->getDataWidth();
		y1  = y2;
		y2 += h1;
		x1 = x2 = 0;
		lastX = -1;
		if (y2 > cy && y1 < cy + h)
		{
			if (y1 > cy)
			{
				painter.setPen(rgbLight);
				painter.drawLine(0, y1 - cy, w, y1 - cy);
			}
			QRect dataRect(0, y1 - cy + 1, w, y2 - y1 - 2);
			painter.fillRect(dataRect, rgbMid);
			painter.setPen(rgbDark);
			painter.drawLine(0, y2 - cy - 1, w, y2 - cy - 1);
			auto item = channel.second.data.begin();
			for(; item != channel.second.data.end(); ++item)
			{
				x1 = mEditor->getPixelFromTime(item->first) - cx;

				// skip items not in view or already drawn
				if (x1 < 0 || x1 == lastX)
					continue;
				lastX = x1;

				// break if we move out of view
				if (x1 > w)
					break;

				painter.setPen(QPen(channel.second.color));
				painter.drawLine(x1, y1 - cy + 1, x1,
				                 y1 - cy + renderer->getIndicatorHeight() - 1);

				if (x1 >= x2 && x1 + w1 <= w)
				{
					QRect rect(x1, y1 - cy + renderer->getIndicatorHeight(),
					           w - x1, h2 - 1);
					QSize s = renderer->renderMessage(&painter, rect,
					                                  channel.second, item);
					x2 = x1 + s.width();
				}
			}
		}
	}

	int64 t = mEditor->getTimeFromPixel(cx);
	t = (t/mEditor->getTimeStep())*mEditor->getTimeStep()-mEditor->getTimeStep();
	int x = mEditor->getPixelFromTime(t)-cx;
	while (x < w)
	{
		if (x>=0)
		{
			painter.setPen(timeIndicator);
			painter.drawLine(x, 0, x, y2 - cy - 2);
		}
		t += mEditor->getTimeStep();
		x = mEditor->getPixelFromTime(t)-cx;
	}

	if (y2 < cy + h)
	{
		painter.setPen(rgbMid);
		painter.drawLine(0, y2 - cy, w, y2 - cy);
	}
}

void TapeDataView::updateContentsHeight()
{
	int contentsHeight = 25 << 1;

	const TapeChannelInfoMap& channels = mEditor->getChannels();
	foreach(const TapeChannelInfoMap::value_type& channel, channels)
		contentsHeight += channel.second.renderer->getTotalHeight();

	resizeContents(contentsWidth(), contentsHeight);
}


void TapeDataView::updateContentsWidth(int contentWidth)
{
	contentWidth += mEditor->getTotalLength();
	resizeContents(contentWidth, contentsHeight());
	mEditor->getTimeView()->resizeContents(contentWidth + 100, mEditor->getTimeView()->contentsHeight());
	mEditor->getTimeView()->updateContents();
}

void TapeDataView::updateContents(const QRect& rect)
{
	updatePixmap(contentsX(), contentsY());
	ScrollView::updateContents(rect);
}

void TapeDataView::updateContents()
{
	updatePixmap(contentsX(), contentsY());
	ScrollView::updateContents();
}

TapeChannelInfo* TapeDataView::getChannelAtPos(const QPoint& pos)
{
	int y1 = 0;
	int y2 = 0;
	TapeChannelInfoMap& channels = mEditor->getChannels();
	foreach(TapeChannelInfoMap::value_type& channel, channels)
	{
		y1  = y2;
		y2 += channel.second.renderer->getTotalHeight();
		if (y2 > pos.y())
			return &channel.second;
	}
	return NULL;
}

TapeChannelInfo::DataMap::iterator TapeDataView::getMessageAtPos(const QPoint& pos,
                                                                 TapeChannelInfo& info)
{
	TapeChannelInfo::DataMap::iterator item = info.data.begin();
	for(; item != info.data.end(); ++item)
	{
		int x1 = mEditor->getPixelFromTime(item->first);
		int x2 = x1 + info.renderer->getDataWidth();
		if (x2 < pos.x())
			continue;
		if (x1 > pos.x())
			return info.data.end();
		return item;
	}
	return info.data.end();
}

void TapeDataView::onContentsMoving(int cx, int cy)
{
	if (contentsY() != cy)
		setContentsPos(contentsX(), cy);
}

void TapeDataView::drawContents(QPainter* painter, const QRect& rect)
{
	painter->drawPixmap(rect, mPixmap, rect);

	int cx = contentsX();
	int x = mEditor->getSelectionStartX() - cx;
	if (x >= rect.left() && x <= rect.right())
	{
		painter->setPen(Qt::green);
		painter->drawLine(x, rect.top(), x, rect.bottom());
	}
	x = mEditor->getSelectionEndX() - cx;
	if (x >= rect.left() && x <= rect.right())
	{
		painter->setPen(Qt::red);
		painter->drawLine(x, rect.top(), x, rect.bottom());
	}
}

bool TapeDataView::eventFilter(QObject* object, QEvent* event)
{
	QWidget* viewport = this->viewport();
	if (static_cast<QWidget*>(object) == viewport)
	{
		if (event->type() == QEvent::ToolTip)
		{
			QHelpEvent* helpEvent = static_cast<QHelpEvent*>(event);
			if (helpEvent)
			{
				QPoint pos = viewportToContents(helpEvent->pos());
				TapeChannelInfo* info = getChannelAtPos(pos);
				if (info != NULL)
				{
					QString txt("<FONT COLOR=black>Time: ");
					TapeChannelInfo::DataMap::iterator item = getMessageAtPos(pos, *info);
					if (item != info->data.end())
					{
						txt += mEditor->getStringFromTime(item->first);
						if (info->renderer->hasFeature(TapeDataRenderer::INFO))
						{
							QString infoTxt = info->renderer->getMessageInfo(*info, item);
							if (!infoTxt.isEmpty())
								txt += "\nInfo: " + infoTxt;
						}
					}
					else
						txt += mEditor->getStringFromTime(mEditor->getTimeFromPixel(pos.x()));
					txt += "</FONT>";
					QToolTip::showText(helpEvent->globalPos(), txt, viewport);
					return true;
				}
			}
		}
	}

	return ScrollView::eventFilter(object, event);
}

void TapeDataView::setSelection(const QRect& rect)
{
	mSelection = rect.normalized();
	int64 start = mEditor->getTimeFromPixel(mSelection.left());
	int64 end = mEditor->getTimeFromPixel(mSelection.right());
	mEditor->setSelectionStart(start);
	mEditor->setSelectionEnd(end);
	mEditor->clearSelectedChannels();
	QRect rubberBandRect = mSelection;
	rubberBandRect.moveTopLeft(contentsToViewport(rubberBandRect.topLeft()));
	mRubberBand->setGeometry(rubberBandRect);
	TapeChannelInfoMap& channels = mEditor->getChannels();
	int y1, y2 = 0;
	foreach(TapeChannelInfoMap::value_type& channel, channels)
	{
		y1 = y2;
		y2 += channel.second.renderer->getTotalHeight();
		if (y1 > mSelection.bottom())
			break;
		if (y2 >= mSelection.top())
			channel.second.selected = true;
	}
	mEditor->getTimeView()->viewport()->update();
	mEditor->getThumbView()->update();
	mEditor->getChannelView()->updateContents();
	viewport()->update();

}

///////////////////////////////////////////////////////////////////////////////

}
