/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DependencyDialog.C
 *    Show dialog with full (install) dependency tree of the package.
 *
 * @author Ronny Stricker
 * @date   2011-09-28
 */

#include "gui/dialogs/DependencyDialog.h"

#include <QPushButton>
#include <QLayout>
#include <QHeaderView>

#include "core/Package.h"

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void recursivePhaseOut( PackageGroup* currentPackage, PackageGroup*& root,
		PackageGroup* currentNew, Database* db )
{
	if ( (!root || !(root->getIdentical( currentPackage )))
			/*&& db->getFlag( currentPackage) != Database::DONOTINSTALL*/  ) {
		PackageGroup* sub = currentPackage->createCopy();
		sub->clearChildren();
		if ( !root ) {
			root = sub;
		}
		else {
			currentNew->addChild( sub );
		}

		Package* subPackage = dynamic_cast<Package*>( sub );

		// add children of groups and children of packages which are
		// supposed to be installed
		if ( ( db->getFlag( currentPackage ) & Database::INSTALL ) ||
			!subPackage ) {
			foreach( PackageGroup* group, currentPackage->mSubPackages ) {
				recursivePhaseOut( group, root, sub, db );
			}
		}
	}
	// else package has been already added
}

Package* phaseOutDependencies( PackageGroup* origDepTree, Database* db )
{
	PackageGroup* group = NULL;
	recursivePhaseOut( origDepTree, group, NULL, db );
	Package* package = dynamic_cast<Package*>( group );
	assert( package );
	return package;
}

///////////////////////////////////////////////////////////////////////////////


DependencyDialog::DependencyDialog( PackageGroup* rootPackage, Database* database,
		QWidget * parent, Qt::WindowFlags f ) : QDialog( parent, f )
{
	QVBoxLayout* layout = new QVBoxLayout( this );

	QLabel* text = new QLabel("The package requires the following packages "
			"to be installed. Select the packages which should be added to "
			"the install plan.",this);
	layout->addWidget( text );

	mTreeView = new QTreeView( this );
	layout->addWidget( mTreeView );

	QLayout* buttonLayout = new QHBoxLayout();
	QPushButton* buttonOK = new QPushButton("&OK",this);
	buttonLayout->addWidget( buttonOK );
	QPushButton* buttonCancel = new QPushButton("&CANCEL",this);
	buttonLayout->addWidget( buttonCancel );
	layout->addLayout( buttonLayout );

	mDepRoot = rootPackage;
	depDatabase = database;
	currentDatabase.repos = database->repos;

	updateDependencies();

	mPackageModel = new PackageTreeModel( mCurrentDep, &currentDatabase,
			PackageTreeModel::EDITABLE );
	mProxyModel = new PackageSortModel(mTreeView);
	mProxyModel->setSortCaseSensitivity(Qt::CaseInsensitive);
	mProxyModel->setSourceModel( mPackageModel );
	mTreeView->setModel( mProxyModel );
	ComboBoxDelegate* delegate = new ComboBoxDelegate(this);
	mTreeView->setItemDelegateForColumn(2,delegate);
	mTreeView->setSortingEnabled(true);

	// NOTE: If mirapackage crashes here, it is most likely caused by a bug
	// in Qt. Until now it crashes only on Fedora12 (Qt-4.6.3). On newer
	// systems (CentOS 6.x, Ubuntu 12.04, SUSE) it works perfectly.
	mTreeView->expandAll();

	mTreeView->resizeColumnToContents( 0 );
	mTreeView->resizeColumnToContents( 1 );
	mTreeView->resizeColumnToContents( 2 );

	connect( mTreeView, SIGNAL(pressed( QModelIndex const& ) ), this,
		SLOT( mousePressed( QModelIndex const& ) ) );

	connect( mTreeView->header(), SIGNAL(sectionClicked ( int )),
	         this, SLOT(headerClicked( int)));

	connect( mPackageModel, SIGNAL(dataChanged(QModelIndex const&,
		QModelIndex const&)), this,SLOT(ascribeChanges(QModelIndex const&)));
	connect( buttonOK, SIGNAL(clicked()), this, SLOT(accept()));
	connect( buttonCancel, SIGNAL(clicked()), this, SLOT(reject()));
}

DependencyDialog::~DependencyDialog()
{
	currentDatabase.repos.clear();
}

void DependencyDialog::updateDependencies()
{
	mCurrentDep = phaseOutDependencies( mDepRoot, depDatabase );
	mCurrentDep->selectStdRepo( depDatabase );

	currentDatabase.mActionPlan.clear();

	typedef std::map<Package*,Database::Action>::value_type MapType;
	foreach( MapType const& flag, depDatabase->mActionPlan ) {
		Package* pkg = dynamic_cast<Package*>( mCurrentDep->getIdentical( flag.first ) );
		if (pkg) {
			currentDatabase.setFlag( pkg, flag.second );
		}
	}
}

void recursiveSetUninstall( PackageGroup* group, Database* db )
{
	Package* package = dynamic_cast<Package*>( group );

	if ( package && db->isInstalled( package, true ) ) {
		db->setFlag(package, Database::NONE );
	}

	foreach( PackageGroup* grp, group->mSubPackages ) {
		recursiveSetUninstall( grp, db );
	}
}

void DependencyDialog::ascribeChanges(QModelIndex const& index)
{
	if ( index.isValid() ) {
		PackageGroup* group = reinterpret_cast<PackageGroup*>( index.internalPointer() );

		// we have to obtain all identical packages since the tree can contain
		// one single package several times
		vector<PackageGroup*> identicalVct;
		mDepRoot->getAllIdentical( group, identicalVct );

		foreach( PackageGroup* grp, identicalVct ) {
			Package* identical = dynamic_cast<Package*>( grp );

			if ( currentDatabase.isInstalled( group, true ) ) {
				depDatabase->recursiveSelectStdSource( identical );
			}
			else {
				// remove all install flags from the children
				recursiveSetUninstall( identical, depDatabase );
			}
		}
	}

	mPackageModel->beginReset();
	updateDependencies();
	mPackageModel->mRoot = mCurrentDep;
	mPackageModel->endReset();
	mTreeView->expandAll();
}

void DependencyDialog::accept()
{
//	 forget the history
//	depDatabase->mInstallFlags.clear();
	// ascribe the changes from the (pruned) internal database to the real database
//	ascribeChanges( QModelIndex() );

	QDialog::accept();
}

void DependencyDialog::mousePressed( QModelIndex const& index )
{
	if ( !index.isValid() || index.column() != 0 )
		return;

	PackageGroup* tGroup = static_cast<PackageGroup*>( mProxyModel->mapToSource(index).internalPointer() );
	Package* tPackage = dynamic_cast<Package*>( tGroup );

	if ( tPackage ) {
		bool install = !currentDatabase.isInstalled( tPackage, true );
		// set changes in depDatabase
		vector<PackageGroup*> identicalVct;
		mDepRoot->getAllIdentical( tPackage, identicalVct );
		foreach( PackageGroup* grp, identicalVct ) {
			Package* identical = dynamic_cast<Package*>( grp );
			assert( identical );
			depDatabase->setFlag( identical, install ? Database::INSTALL : Database::DONOTINSTALL );
		}
		// set changed in the currentDatabase
		mPackageModel->setData(mProxyModel->mapToSource(index), install, Qt::CheckStateRole);
	}
}

void DependencyDialog::headerClicked( int column )
{
	if ( column == 0 )
		mProxyModel->setSortRole( Qt::CheckStateRole );
	else
		mProxyModel->setSortRole( Qt::DisplayRole );
	mTreeView->expandAll();
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

