/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LocalRepository.C
 *    Description.
 *
 * @author Christian Martin
 * @date   2012/10/07
 */

#include "core/LocalRepository.h"

#include <utils/PathFinder.h>
#include <serialization/Serialization.h>

#include "core/PackageParser.h"

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

LocalRepository::LocalRepository(QObject* parent) :
	FileRepository(parent)
{
	protocol = "local";
	name = getClass().getMetaInfo("RepoType");
	url += name;
}

void LocalRepository::setUrl(const Url& iUrl)
{
	Url tUrl = url;
	try {
		FileRepository::setUrl(iUrl);
	} catch(Exception &ex) {
		url = tUrl;
		MIRA_RETHROW(ex, "");
	}

	int colonIdx = url.find(":");
	string protocol = url.substr(0, colonIdx);

	if (protocol != "file") {
		MIRA_THROW(XLogical, "Don't know how to handle protocol " << protocol);
	}

	// Extract the repository root path from the Url
	// Format of url: "file:///path" => We have to skip 3 characters.
	mRootPath = Path(url.substr(colonIdx+3));
	if (!boost::filesystem::is_directory(mRootPath)) {
		MIRA_LOG(WARNING) << "The path " << mRootPath << " does not exist!";
	}
}

void LocalRepository::examine(std::vector<Package*>& oPackages, bool thorough,
                              boost::function<void (std::string const&,int,int)> progressListener)
{
	// Get the list of interesting files
	PathVector pkgFiles = findFiles(mRootPath, "*.(package|extpackage)", true);
	PathVector mountFiles = findFiles(mRootPath, "mountdir.xml", true);
	int progress = 0;
	int totalFileCnt = pkgFiles.size() + mountFiles.size();

	// Parse all mountdir.xml
	foreach(Path p, mountFiles) {
		Path relPath = relativizePath(p, mRootPath);
		Url relUrl(relPath.string());
		std::string fileContent = catFile(Url(relPath.string()));
		parseMountDirXml(url/relUrl, fileContent);
		++progress;
		if ( progressListener )
			progressListener(relUrl, progress, totalFileCnt);
	}

	// Parse all package files
	foreach(Path p, pkgFiles) {
		Path relPath = relativizePath(p, mRootPath);
		Url relUrl(relPath.string());
		Package* tPackage = indexWebPackage(relUrl, this);
		if (tPackage)
			oPackages.push_back(tPackage);
		++progress;
		if (progressListener)
			progressListener(relUrl, progress, totalFileCnt);
	}
}

std::vector<Url> LocalRepository::getZipFileList(const Url& subdir)
{
	std::vector<Url> tReturn;
	PathVector zipFiles = findFiles(mRootPath / subdir, "*.zip", true);
	foreach(Path p, zipFiles) {
		Path relPath = relativizePath(p, mRootPath);
		Url relUrl(relPath.string());
		tReturn.push_back(relUrl);
	}
	return tReturn;
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::LocalRepository, mira::FileRepository);
