/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RigidModel.h
 *    A rigid model representation.
 *
 * @author Tim Langner
 * @date   2011/05/09
 */

#ifndef _MIRA_RIGIDMODEL_H_
#define _MIRA_RIGIDMODEL_H_

#include <fw/Framework.h>

#include <serialization/Serialization.h>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/list>
#include <serialization/adapters/boost/shared_ptr.hpp>

#include <model/Link.h>
#include <model/Joint.h>
#include <model/Footprint.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

/**
 * Class representing a named rigid model. A rigid model consists of:
 * - \ref mira::model::Link "Links" - A link is a single body part of the model.
 * - \ref mira::model::Joint "Joints" - A joint connects two body parts (links)
 * - \ref mira::model::Material "Materials" - A material can be used when visualizing a rigid model
 *   two links.
 */
class MIRA_RIGID_MODEL_EXPORT RigidModel : public Object
{
	MIRA_OBJECT(RigidModel)
public:
	
	/// A list of links
	typedef std::list<LinkPtr> LinkList;
	/// Maps joint pointers to their name
	typedef std::map<std::string, JointPtr> JointMap;
	/// Maps material pointers to their name
	typedef std::map<std::string, MaterialPtr> MaterialMap;

	/// Destructor
	virtual ~RigidModel() {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Name", name, "The name of this model", "");
		r.member("Links", links, "The links of this model", LinkList());
		r.member("Joints", joints, "The joints of this model", JointMap());
		r.member("Materials", materials, "The materials of this model", MaterialMap());
	}

	/**
	 * Specialized reflect for XMLDeserializer.
	 */
	void reflect(XMLDeserializer& r);

	/**
	 * Clears the rigid model.
	 */
	void clear();

	/**
	 * Returns the name of the root link, the only link that has no parent.
	 * @throw XLogical, if NONE or MORE than one root link exists
	 */
	std::string getRootLink() const;

	/**
	 * List of collision geometries, each item consisting of the geometry
	 * pointer and a transform that describes the position of the geometry.
	 */
	typedef std::list<std::pair<GeometryPtr, RigidTransform3f>,
	                  Eigen::aligned_allocator<std::pair<GeometryPtr, RigidTransform3f>>> CollisionGeometries;

	/**
	 * Returns a list of the models collision geometries and the corresponding
	 * transforms for each geometry that describes its position within the
	 * model.
	 * All transforms are given in relation to the specified
	 * target frame. If the target frame is left blank, the root link of the
	 * model is used, i.e. the footprint is given in the coordinate frame of the
	 * model.
	 * The optional timestamp parameter allows to specify a timestamp that is
	 * used for querying the transforms.
	 * @param[in] targetFrame All transforms will be given relative to this frame.
	 *            If empty (default) the root frame of the model will be used,
	 *            so all transforms will be relative to the models origin.
	 * @param[in] timestamp Timestamp used for querying transforms
	 * @param[in] filter Filter for filtering only collision geometries where the name matches the filter
	 *            By default collision geometries with no name are returned
	 */
	CollisionGeometries getCollisionGeometries(std::string targetFrame="",
	                                           const Time& timestamp = Time::now(),
	                                           const std::string& filter = "") const;

	/**
	 * Returns a list of the bounding boxes for each of the models collision
	 * geometries at their correct positions.
	 * All transforms and hence the position of the boxes are given in relation
	 * to the specified target frame. If the target frame is left blank,
	 * the root link of the model is used, i.e. the bounding boxes are given in the
	 * coordinate frame of the model.
	 * The optional timestamp parameter allows to specify a timestamp that is
	 * used for querying the transforms.
	 * @param[in] targetFrame All box positions will be given relative to this frame.
	 *            If empty (default) the root frame of the model will be used,
	 *            so all box positions will be relative to the models origin.
	 * @param[in] timestamp Timestamp used for querying transforms
	 * @param[in] filter Filter for filtering only collision geometries where the name matches the filter
	 *            By default collision geometries with no name are used for calculating the bounding boxes
	 */
	std::list<Box3f> getCollisionBoundingBoxes(std::string targetFrame="",
	                                           const Time& timestamp = Time::now(),
	                                           const std::string& filter = "") const;

	/**
	 * Returns the models footprint (the projection on the xy-plane).
	 * The geometry of the footprint is returned in relation to the specified
	 * target frame. If the target frame is left blank, the root link of the
	 * model is used, i.e. the footprint is given in the coordinate frame of the
	 * model.
	 * @param[in] targetFrame The footprint will be given relative to this frame.
	 *            If empty (default) the root frame of the model will be used,
	 *            so the footprint will be relative to the models origin.
	 * @param[in] timestamp Timestamp used for querying transforms
	 * @param[in] filter Filter for filtering only collision geometries where the name matches the filter
	 *            By default collision geometries with no name are used for calculating the footprint
	 */
	Footprint getFootprint(std::string targetFrame = "",
	                       const Time& timestamp = Time::now(),
	                       const std::string& filter = "") const;

	/**
	 * Resolves all link names in the model and adds the transforms between them
	 * to the transform tree.
	 * @param ns The namespace of the model.
	 * @param timestamp An optional timestamp of the (initial) transformation states
	 */
	void resolveAndPublishLinks(const std::string& ns,
	                            const Time& timestamp = Time::now());

	/**
	 * Publishes the state of the joint with the given name. Can be used for 
	 * CONTINUOUS, REVOLUTE and PRISMATIC joints.
	 * @throw XInvalidConfig if used with not supported types of joints
	 * @param name Name of the joint to alter
	 * @param value The new value for the joint
	 *        - radian for CONTINUOUS and REVOLUTE
	 *        - meter for PRISMATIC
	 * @param timestamp An optional timestamp of the new joint state
	 */
	void publishJoint(const std::string& name, float value,
	                  const Time& timestamp = Time::now()) const;

	std::string name;		///< Name of the model
	LinkList links;			///< List of links used by this model
	JointMap joints;		///< Map with name - joint pointer pairs used by this model
	MaterialMap materials;	///< Map with name - material pointer pairs used by this model
};

typedef boost::shared_ptr<RigidModel> RigidModelPtr;

///////////////////////////////////////////////////////////////////////////////

}}

#endif
