/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PositionTask.h
 *    Positioning task with a given position and a variance.
 *
 * @author Tim Langner
 * @date   2011/04/18
 */

#ifndef _MIRA_POSITIONTASK_H_
#define _MIRA_POSITIONTASK_H_

#include <geometry/Point.h>

#include <navigation/Task.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Task for driving to a given position providing two limits for a hysteresis
 * that can be applied when checking for goal reached.
 * The minimum tolerance is the distance where the path planner stops driving
 * nearer to the given position.
 * The maximum tolerance is the maximum allowed distance to the given position.
 * When this distance is exceeded the path planner starts again driving nearer
 * to the position until the minimum distance is reached again....
 */
class MIRA_NAVIGATION_EXPORT PositionTask : public SubTask
{
	MIRA_OBJECT(PositionTask);
public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	/// Creates a task with both tolerances 0
	PositionTask() :
		position(Point2f(0.0, 0.0)),
		minTolerance(0.0f),
		maxTolerance(0.0f),
		frameID("/GlobalFrame")
	{}

	/**
	 * Creates a task with given position and one tolerance that is used for
	 * both min and max tolerance.
	 * @param[in] pos  The request position.
	 * @param[in] maxT The min and max tolerance in [m].
	 * @param[in] f    The frame ID.
	 */
	PositionTask(const Point2f& pos, float maxT,
	             const std::string& f = "/GlobalFrame") :
		position(pos),
		minTolerance(maxT),
		maxTolerance(maxT),
		frameID(f)
	{}

	/**
	 * @brief Creates a task with given position and tolerances
	 * @param[in] pos  The request position.
	 * @param[in] minT The minimum tolerance in [m].
	 * @param[in] maxT The maximum tolerance in [m].
	 * @param[in] f    The frame ID.
	 */
	PositionTask(const Point2f& pos, float minT, float maxT,
	             const std::string& f = "/GlobalFrame") :
		position(pos),
		minTolerance(minT<maxT?minT:maxT),
		maxTolerance(maxT>minT?maxT:minT),
		frameID(f)
	{}

	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, SubTask);
		r.property("Position", position,
		           "The request position.");
		r.property("MinTolerance", minTolerance,
		           "The minimum tolerance/lower limit for path planner hysteresis.");
		r.property("MaxTolerance", maxTolerance,
		           "The maximum tolerance/upper limit for path planner hysteresis.");
		r.property("FrameID", frameID,
		           "The frame the position is related to.", "/GlobalFrame");
	}

	/// Calculates if the task is reached for the current position pos
	bool reached(const Point2f& pos) const
	{
		return hypotf(position.x()-pos.x(),
		              position.y()-pos.y()) <= maxTolerance;
	}

public:
	/// The request position.
	Point2f position;

	/// The minimum tolerance/lower limit for path planner hysteresis.
	float minTolerance;

	/// The maximum tolerance/upper limit for path planner hysteresis.
	float maxTolerance;

	/// The frame the position is related to
	std::string frameID;
};

///////////////////////////////////////////////////////////////////////////////

}}

#endif
