/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LocalizeTool.C
 *    Tool plugin for miracenter to localize a robot in an environment using
 *    the ILocalization service interface.
 *
 * @author Erik Einhorn
 * @date   2011/07/02
 */

#include <fw/ServiceProperty.h>

#include <visualization/3d/PoseTool.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Tool plugin for miracenter to localize a robot in an environment using the
 * ILocalization service interface.
 */
class LocalizeToolBase : public PoseTool
{

public:

	LocalizeToolBase() :
		mLocalizationService("ILocalization")
	{}

	LocalizeToolBase(const PoseTool::InputModePosition& mode) :
		PoseTool(mode), mLocalizationService("ILocalization")
	{}

	LocalizeToolBase(const PoseTool::InputModeOrientation& mode) :
		PoseTool(mode), mLocalizationService("ILocalization")
	{}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, PoseTool);
		r.property("LocalizationService", mLocalizationService, "Service used for localization",
		           "");

	}

	/** @name Implementation of VisualizationTool */
	//@{

	virtual QIcon getIcon() {
		QPixmap pixmap(":/icons/ToolLocalize.png");
		return QIcon(pixmap);
	}

	//@}

protected:

	std::string getService() {
		std::string service = mLocalizationService;
		// user did not provide a service -> search for interface
		if (service.empty())
		{
			auto l = queryServicesForInterface("ILocalization");
			if (!l.empty())
				service = l.front();
		}
		return service;
	}

	ServiceProperty mLocalizationService;
};

class LocalizeTool : public LocalizeToolBase
{
	MIRA_META_OBJECT(LocalizeTool,
		("Name", "Localize")
		("Category", "Localization")
		("Description", "Set pose for localization")
		("Default", "true")
		("Order", "100"))

protected:
	/** @name Implementation of PoseTool */
	//@{

	virtual void onNewPose(const PoseCov2& pose)
	{
		std::string service = getService();
		if (!service.empty()) {
			// cov actually designates 2sigma regions
			PoseCov2 pose2 = pose;
			pose2.cov = (0.5f * pose.cov).cwiseAbs2();
			RPCFuture<void> result = callService<void>(service, "setInitPose", pose2,
			                                           getSite()->getFixedFrame());
			result.timedWait(Duration::seconds(10));
			result.get(); // will cause exception if something went wrong
		}
	}

	//@}
};

class LocalizePositionTool : public LocalizeToolBase
{
	MIRA_META_OBJECT(LocalizePositionTool,
		("Name", "Localize - Position only")
		("Category", "Localization")
		("Description", "Set position for localization")
		("Default", "false")
		("Order", "101"))

public:

	LocalizePositionTool() : LocalizeToolBase(PoseTool::InputModePosition()) {}

protected:
	/** @name Implementation of PoseTool */
	//@{

	virtual void onNewPosition(const Point2f& position,
	                           float xRange, float yRange)
	{
		std::string service = getService();
		if (!service.empty()) {
			// ranges designate 2sigma regions
			Eigen::Matrix<float,2,2> cov = Eigen::Matrix<float,2,2>::Zero();
			cov(0, 0) = 0.25f * xRange * xRange;
			cov(1, 1) = 0.25f * yRange * yRange;
			RPCFuture<void> result = callService<void>(service, "setInitPosition",
			                                           position, cov,
			                                           getSite()->getFixedFrame());
			result.timedWait(Duration::seconds(10));
			result.get(); // will cause exception if something went wrong
		}
	}

	//@}
};

class LocalizeOrientationTool : public LocalizeToolBase
{
	MIRA_META_OBJECT(LocalizeOrientationTool,
		("Name", "Localize - Orientation only")
		("Category", "Localization")
		("Description", "Set orientation for localization")
		("Default", "false")
		("Order", "102"))

public:

	LocalizeOrientationTool() : LocalizeToolBase(PoseTool::InputModeOrientation()) {}

protected:
	/** @name Implementation of PoseTool */
	//@{

	virtual void onNewOrientation(const Anglef& orientation,
	                              float phiRange)
	{
		std::string service = getService();
		if (!service.empty()) {
			// cov actually designates 2sigma regions
			RPCFuture<void> result = callService<void>(service, "setInitOrientation",
			                                           orientation, 0.25f * phiRange * phiRange,
			                                           getSite()->getFixedFrame());
			result.timedWait(Duration::seconds(10));
			result.get(); // will cause exception if something went wrong
		}
	}

	//@}
};
///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::navigation::LocalizeTool, mira::VisualizationTool3D)
MIRA_CLASS_SERIALIZATION(mira::navigation::LocalizePositionTool, mira::VisualizationTool3D)
MIRA_CLASS_SERIALIZATION(mira::navigation::LocalizeOrientationTool, mira::VisualizationTool3D)
