/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GroundPlaneVisualization.C
 *    Visualization of the ground plane.
 *
 * @author Erik Einhorn
 * @date   2011/01/02
 */

#include <serialization/Serialization.h>

#include <visualization/Visualization3D.h>
#include <visualization/3d/GridObject.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <widgets/OgreUtils.h>
#include <OGRE/OgreSceneManager.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

/**
 * A ground plane visualization.
 */
class GroundPlaneVisualization3D :  public Visualization3D
{
	MIRA_META_OBJECT(GroundPlaneVisualization3D,
		("Name", "Ground Plane")
		("Description", "Shows a ground plane in the xy-plane"))

public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	GroundPlaneVisualization3D() :
		mGrid(NULL),
		mColor(Ogre::ColourValue::Black),
		mCellCount(20),
		mCellSize(0.2f),
		mOffsetX(0.0f),
		mOffsetY(0.0f)
	{
	}

	/// The destructor.
	virtual ~GroundPlaneVisualization3D() {
		delete mGrid;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3D);
		r.property("CellCount", mCellCount,
		           setter<int>(&GroundPlaneVisualization3D::setCellCount, this),
		           "The number of cells", 20);
		r.property("CellSize", mCellSize,
		           setter<float>(&GroundPlaneVisualization3D::setCellSize, this),
		           "The size of a cell in meter", 0.2f);
		r.property("OffsetX", mOffsetX,
		           setter<float>(&GroundPlaneVisualization3D::setOffsetX, this),
		           "The x-offset of the grid in meter", 0.0f);
		r.property("OffsetY", mOffsetY,
		           setter<float>(&GroundPlaneVisualization3D::setOffsetY, this),
		           "The x-offset of the grid in meter", 0.0f);
		r.property("Color", mColor,
		           setter<Ogre::ColourValue>(&GroundPlaneVisualization3D::setColor, this),
		           "The color of the grid", Ogre::ColourValue::Black);
	}

	//@}

public:
	/** @name Setters */
	//@{

	/// Set the number of cells.
	void setCellCount(int count)
	{
		mCellCount = count;
		if (mGrid != NULL)
			mGrid->setCellCount(mCellCount);
	}

	/// Set the cell size in [m].
	void setCellSize(float size)
	{
		mCellSize = size;
		if (mGrid != NULL)
			mGrid->setCellSize(mCellSize);
	}

	/// Set the x-offset in [m].
	void setOffsetX(float offsetX)
	{
		mOffsetX = offsetX;
		if (mGrid != NULL)
			mGrid->setTransform(Pose2(mOffsetX, mOffsetY, 0.0f));
	}

	/// Set the y-offset in [m].
	void setOffsetY(float offsetY)
	{
		mOffsetY = offsetY;
		if (mGrid != NULL)
			mGrid->setTransform(Pose2(mOffsetX, mOffsetY, 0.0f));
	}

	/// Set the color of the grid.
	void setColor(const Ogre::ColourValue& c)
	{
		mColor = c;
		if (mGrid != NULL)
			mGrid->setColor(mColor);
	}

	//@}

public:
	/** @name Public implementation of Visualization3D and Visualization*/
	//@{

	virtual void setupScene(IVisualization3DSite* site)
	{
		Ogre::SceneManager* mgr = site->getSceneManager();
		mGrid = new GridObject(mgr);
		mGrid->setCellCount(mCellCount);
		mGrid->setCellSize(mCellSize);
		mGrid->setColor(mColor);
		mGrid->setVisible(isEnabled());
		mGrid->setTransform(Pose2(mOffsetX, mOffsetY, 0.0f));
	}

	virtual Ogre::SceneNode* getNode()
	{
		if (!mGrid)
			return NULL;

		return mGrid->getNode();
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3D::setEnabled(enabled);
		if (mGrid != NULL)
			mGrid->setVisible(enabled);
	}

	//@}

private:
	GridObject* mGrid;
	Ogre::ColourValue mColor;
	int mCellCount;
	float mCellSize;
	float mOffsetX;
	float mOffsetY;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::GroundPlaneVisualization3D,
                         mira::Visualization3D);
