/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JpegCodec.C
 *    Implementation of JpegCodec.h.
 *
 * @author Erik Einhorn
 * @date   2011/09/29
 */

#include <codec/JpegCodec.h>
#include <opencv2/opencv.hpp>

#include <serialization/Serialization.h>

#include <utils/TypedVoidPtr.h>

namespace mira { namespace codec {

///////////////////////////////////////////////////////////////////////////////

TypeId JpegCodec::getSupportedTypeId() const
{
	return typeId<cv::Mat>();
}

Buffer<uint8> JpegCodec::encodeBuffer(TypedVoidConstPtr objectPtr)
{
	std::vector<int> params;
#if CV_MAJOR_VERSION >= 4
	params.push_back(cv::IMWRITE_JPEG_QUALITY);
#else
	params.push_back(CV_IMWRITE_JPEG_QUALITY);
#endif
	params.push_back(mQuality);

	// get object from void pointer
	const cv::Mat* image = objectPtr;

	if ((image->rows == 0) || (image->cols == 0))
		MIRA_THROW(XIO, "Trying to encode empty image")

	std::vector<uint8> compressedData;
	if(!cv::imencode(".jpg", *image, compressedData, params))
		MIRA_THROW(XIO, "Failed to encode data in JpegCodec");

	// move content of vector into buffer
	Buffer<uint8> buf(std::move(compressedData));

	return std::move(buf);
}

void JpegCodec::decodeBuffer(const Buffer<uint8>& data, TypedVoidPtr ioObjectPtr)
{

	cv::Mat buf(cv::Size(1,data.size()), CV_8UC1, (void*)data.data());
	// get object from void pointer
	cv::Mat* image = ioObjectPtr;

	*image = cv::imdecode(buf, -1);
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::codec::JpegCodec, mira::BinarySerializerCodec);
