/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OmniCameraIntrinsic.h
 *    Normalized and unnormalized intrinsic parameters for Omni cameras
 *
 * @author Michael Volkhardt
 * @date   2011/09/22
 */

#ifndef _MIRA_CAMERA_OMNICAMERAINTRINSIC_H_
#define _MIRA_CAMERA_OMNICAMERAINTRINSIC_H_

#include <vector>

#include <math/Eigen.h>
#include <geometry/Size.h>

#include <cameraparameters/CameraParametersExports.h>

#include <opencv2/core/version.hpp>

namespace mira {
namespace camera {

//////////////////////////////////////////////////////////////////////////////

/**
 * Describes the normalized intrinsic parameters of an Omni camera from the OCamToolbox.
 * Also works for Fisheye camera up to 150°.
 * For each physical camera just one instance of this class must be used since
 * the intrinsic camera parameters remain constant.
 *
 * The centerX and centerY parameters must be specified relative to the image
 * extension and not in pixel, i.e. must be passed as normalized parameter
 * centerX / image.width().
 *
 * Furthermore K describes the parameters of a polynom with degree K.size() to
 * convert the pixels into world coordinates. The K_inv vector is an approximation
 * of the inverse polynom parameters used to transform world coordinates
 * into image coordinates.
 *
 * The Matrix A describes the missalignment of the camera center to the mirror
 * center.
 *
 * To obtain those parameters use the matlab toolbox: OCamCalib
 *
 */
class MIRA_CAMERAPARAMETERS_EXPORT OmniCameraIntrinsicNormalized {
public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	/// Constructor.
	OmniCameraIntrinsicNormalized();

	/**
	 * Constructor that sets all intrinsic parameters
	 */
	OmniCameraIntrinsicNormalized(float iCenterX, float iCenterY,
	                              std::vector<float> iPolynomParam,
	                              std::vector<float> iInversePolyParam,
	                              Eigen::Matrix2f const& A);

public:

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.property("CenterX", centerX, "normalized image center x");
		r.property("CenterY", centerY, "normalized image center y");
		r.property(
				"PolynomParam",
				K,
				"polynomial coefficients of transform function (to unit sphere)");
		r.property(
				"InversePolynomParam",
				K_inv,
				"polynomial coefficients of inverse transform function (from unit sphere)");
		r.property("A", A, "parameter of camera-mirror offset A(c,d;e,1)");
	}

public:

	float centerX; ///< normalized image centerX
	float centerY; ///< normalized image centerY
	std::vector<float> K; ///< model parameters
	std::vector<float> K_inv; ///< inverse model parameters

	/// offset of camera center to mirror center using matrix A[c,d;e,1]
	Eigen::Matrix2f A;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Describes the intrinsic parameters of an Omni camera from the OCamToolbox.
 * Also works for Fisheye camera up to 150°.
 * For each physical camera just one instance of this class must be used since
 * the intrinsic camera parameters remain constant.
 *
 * The centerX and centerY parameters are specified relative to the image in pixel,
 * i.e. must be passed as the true image center.
 *
 * Furthermore K describes the parameters of a polynom with degree K.size() to
 * convert the pixels into world coordinates. The K_inv vector is an approximation
 * of the inverse polynom parameters used to transform world coordinates
 * into image coordinates.
 *
 * The Matrix A describes the missalignment of the camera center to the mirror
 * center.
 *
 * To obtain those parameters use the matlab toolbox: OCamCalib
 *
 */
class MIRA_CAMERAPARAMETERS_EXPORT OmniCameraIntrinsic {
public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	/// Constructor.
	OmniCameraIntrinsic();

	/**
	 * constructor that sets all intrinsic parameters
	 */
	OmniCameraIntrinsic(float iCenterX, float iCenterY,
	                    std::vector<float> const& iPolynomParam,
	                    std::vector<float> const& iInversePolyParam,
	                    Eigen::Matrix2f const& A);
	/**
	 * constructor that takes a normalized parameter class to and multiplies
	 * normalized parameters (centerX and centerY)  with the given image size
	 */
	OmniCameraIntrinsic(
			const OmniCameraIntrinsicNormalized& iNormalizedIntrinsic,
			const Size2i& iImageSize);

public:

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.property("CenterX", centerX, "image center x in pixel");
		r.property("CenterY", centerY, "image center y in pixel");
		r.property(
				"PolynomParam",
				K,
				"polynomial coefficients of transform function (to unit sphere)");
		r.property(
				"InversePolynomParam",
				K_inv,
				"polynomial coefficients of inverse transform function (from unit sphere)");
		r.property("A", A, "parameters of camera-mirror offset");
	}

public:

	float centerX; ///< image centerX in pixel
	float centerY; ///< image centerY in pixel
	std::vector<float> K; ///<model parameters
	std::vector<float> K_inv;
	/// offset of camera center to mirror center using matrix A[c,d;e,1]
	Eigen::Matrix2f A;

};

///////////////////////////////////////////////////////////////////////////////
#if CV_MAJOR_VERSION >= 3
/**
 * Describes the intrinsic parameters of an fisheye camera from the OpenCV Camera Calibration.
 * For each physical camera just one instance of this class must be used since
 * the intrinsic camera parameters remain constant.
 *
 * The centerX and centerY parameters must be specified relative to the image
 * extension and not in pixel, i.e. must be passed as normalized parameter
 * centerX / image.width().
 *
 * Furthermore K the camera matrix containing the real image center and the focal distances.
 *
 * The Matrix D contains the four distortion coefficents k1, k2, k3, k4.
 *
 * To obtain those parameters use the CameraCalibration contributed in OpenCV-Version 3.1.0
 *
 */
class MIRA_CAMERAPARAMETERS_EXPORT FisheyeCameraIntrinsicNormalized {
public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	/// Constructor.
	FisheyeCameraIntrinsicNormalized();

	/**
	 * Constructor that sets all intrinsic parameters
	 */
	FisheyeCameraIntrinsicNormalized(cv::Matx33d K, cv::Vec4d D);

public:

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.property("CameraMatrix",	K, "camera matrix containing fx, fy, cx, cy");
		r.property("D", D, "distortion coefficients");
	}

public:
	cv::Matx33d K; 	///< 3x3 floating-point camera matrix
	cv::Vec4d D;	///< distortion coefficients

};

///////////////////////////////////////////////////////////////////////////////

/**
 * Describes the intrinsic parameters of an fisheye camera from the OpenCV Camera Calibration.
 * For each physical camera just one instance of this class must be used since
 * the intrinsic camera parameters remain constant.
 *
 * Furthermore K the camera matrix containing the real image center and the focal distances.
 *
 * The Matrix D contains the four distortion coefficents k1, k2, k3, k4.
 *
 * To obtain those parameters use the CameraCalibration contributed in OpenCV-Version 3.1.0
 *
 */
class MIRA_CAMERAPARAMETERS_EXPORT FisheyeCameraIntrinsic {
public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	/// Constructor.
	FisheyeCameraIntrinsic();

	/**
	 * constructor that sets all intrinsic parameters
	 */
	FisheyeCameraIntrinsic(cv::Matx33d K, cv::Vec4d D);
	/**
	 * constructor that takes a normalized parameter class to and multiplies
	 * normalized parameters (centerX and centerY)  with the given image size
	 */
	FisheyeCameraIntrinsic(
			const FisheyeCameraIntrinsicNormalized& iNormalizedIntrinsic,
			const Size2i& iImageSize);

public:

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.property("CameraMatrix",	K, "camera matrix containing fx, fy, cx, cy");
		r.property("D", D, "distortion coefficients");
	}

public:

	cv::Matx33d K; 	///< 3x3 floating-point camera matrix
	cv::Vec4d D;	///< distortion coefficients

};
///////////////////////////////////////////////////////////////////////////////
#endif
}
} // namespace

#endif
