###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Provides MIRA_DIST_LIBRARY, MIRA_DIST_BINARY, MIRA_DIST_FILE and MIRA_DIST_FILES
# a macro that install the created library, binary or file(s) to the <somewhere>.
# On Linux the library/binary is installed as a symbolic link, on Window the
# library/binary is copied.
#
# Usage:
#   MIRA_DIST_LIBRARY(TARGET [subdir])
#   MIRA_DIST_BINARY(TARGET [subdir])
#   MIRA_DIST_FILE(TARGET source [subdir])
#   MIRA_DIST_FILES(TARGET
#                   [FILES   <file1 ... fileN>]
#                   [SUBDIR  <subdir>]
#                  )
#
# The possible subdirs are defined in ALLOWED_LIB_SUBDIRS and ALLOWED_BIN_SUBDIRS.
# - For libs they are as follows: units loaders plugins gui ORGE
# - For bin they are as follows: tools
#
# Author: Erik Einhorn, Christian Martin
#
###############################################################################

# Generic macro for creating a symbolic link (on Windows creating a copy) from
# sourcePath to destPath. The commands for doing so are assigned to the 
# specified target. Hence, the symbolic link/copy is created when the target
# is built. On Linux, the symbolic link even is created, if the target is up
# to date (this is important for smooth transition between debug and release 
# builds)
# This macro is used by MIRA_ADD_LIBRARY, MIRA_ADD_BINARY, ADD_MANIFEST, ...
#
macro(MIRA_DIST_FILE_EX target sourcePath destPath)

	# clean the source and dest paths (remove double // and \\)
	get_filename_component(srcDir ${sourcePath} PATH)
	get_filename_component(srcFile ${sourcePath} NAME)
	set(sourcePath2 "${srcDir}/${srcFile}")

	get_filename_component(destDir ${destPath} PATH)
	get_filename_component(destFile ${destPath} NAME)
	set(destPath2 "${destDir}/${destFile}")

	# generate a relative path between the two absolute paths
	file(RELATIVE_PATH relativeFromSourceToDest ${destDir} ${sourcePath2})

	#for debugging:
	#message("sourcePath = " ${sourcePath2})
	#message("destPath   = " ${destPath2})
	#message("destDir    = " ${destDir})
	#message("relativeFromSourceToDest = " ${relativeFromSourceToDest})

	# Now add commands for 'installing' the file. We have to handle Windows
	# and Linux differently:
	# 
	# On Linux:
	#  - Create a separate xyz_dist target, which will trigger the commands
	#    (this ensures that we can switch seamlessly between release and debug
	#     builds, and it ensures that the files/links are refreshed even if the
	#     target is up to date)
	#  - Use symbolic links
	#  - the actual build target xyz depends on xyz_dist, i.e. the symlink is
	#    created before the target is built, which is okay, as Linux allows to
	#    create symlinks to targets even if the target file does not exist yet. 
	#
	# On windows:
	#  - Use POST_BUILD to execute the commands after the target was built
	#  - Copy the files to the destination
	#  - the xyz_dist target depends on the actual build target, i.e. the files
	#    are copied after the target is built
	#


	if(WIN32)
	
		# Although it is not associated with a command on windows, we do need the
		# dist target in order to "trigger" the manifest target
		# target_dist->target : target_dist (copying) is processed AFTER building the target
		if(NOT TARGET ${target}_dist)
			add_custom_target(${target}_dist ALL)
			add_dependencies(${target}_dist ${target})
		endif()

		# Add a custom command to the target, which copies the file.
		# Also see ticket #296 !
		add_custom_command(TARGET ${target} POST_BUILD
			#COMMAND ${CMAKE_COMMAND} -E make_directory "${destDir}"
			#COMMAND ${CMAKE_COMMAND} -E remove "${destPath2}"
			#COMMAND ${CMAKE_COMMAND} -E copy "${sourcePath2}" "${destPath2}"
			COMMAND ${CMAKE_COMMAND}
				-D "COPY_SOURCE_PATH=${sourcePath2}"
				-D "COPY_DEST_PATH=${destPath2}"
				-P "${MIRA_ROOT_DIR}/make/CopyFileIfExists.cmake"
		)
	else() # LINUX
		# target->target_dist : target_dist (creating the symlink) is processed BEFORE target
		if(NOT TARGET ${target}_dist)
			add_custom_target(${target}_dist ALL)
			add_dependencies(${target} ${target}_dist)
		endif()
		add_custom_command(TARGET ${target}_dist
			COMMAND ${CMAKE_COMMAND} -E make_directory "${destDir}"
			COMMAND ${CMAKE_COMMAND} -E remove "${destPath2}"
			COMMAND ${CMAKE_COMMAND} -E create_symlink "${relativeFromSourceToDest}" "${destPath2}"
		)
	endif()

endmacro(MIRA_DIST_FILE_EX)

##############################################################################

# the allowed subdirs
set(ALLOWED_LIB_SUBDIRS units loaders plugins gui OGRE)

# the root lib dir (for .lib files)
set(LIB_DIST_ROOT_DIR ${CMAKE_SOURCE_DIR}/lib)

# the root dll dir (for .so/.dll files)
if (WIN32)
	set(DLL_DIST_ROOT_DIR ${CMAKE_SOURCE_DIR}/bin)
else(WIN32) # LINUX
	set(DLL_DIST_ROOT_DIR ${LIB_DIST_ROOT_DIR})
endif(WIN32)

###############################################################################

macro(DEFINE_EXPORT_SYMBOL target)
	if(WIN32)
		string(TOUPPER "${target}" capitalTarget)
		set_target_properties(${target} PROPERTIES DEFINE_SYMBOL "${capitalTarget}_EXPORTS")
	endif(WIN32) 
endmacro(DEFINE_EXPORT_SYMBOL)

###############################################################################

macro(MIRA_DIST_LIBRARY_EX target targetPath)
	set(subdir ${ARGV2}) # optional 2nd parameter that specifies the subdirectory

	#check if 'subdir' is in the list of allowed subdirectories (but only if we have a subdir)
	if(${ARGC} GREATER 2)
		list(FIND ALLOWED_LIB_SUBDIRS ${subdir} found)
		if(${found} EQUAL -1)
			message("found = " ${found})
			message(FATAL_ERROR "The following subdirectories are allowed for MIRA_DIST_LIBRARY only: " "${ALLOWED_LIB_SUBDIRS}")
		endif()
	endif()

	# extract the library filename
	get_filename_component(targetFilename ${targetPath} NAME)

	# build the full path where we should put the symbolic link
	set(installPath ${DLL_DIST_ROOT_DIR}/${subdir}/${targetFilename})

	# build dirs for ".LIB" file
	get_filename_component(targetDir ${targetPath} PATH)
	get_filename_component(targetFilenameWe  ${targetPath} NAME_WE)
	set(targetFilenameLib "${targetFilenameWe}.lib")
	set(targetPathLib "${targetDir}/${targetFilenameLib}")
	set(installPathLib ${LIB_DIST_ROOT_DIR}/${subdir}/${targetFilenameLib})

	# get values of VERSION and SOVERSION
	get_target_property(targetVersion ${target} VERSION)
	get_target_property(targetSOVersion ${target} SOVERSION)

	# for debugging:
	#message("subdir = " ${subdir})
	#message("fullOutputPath = " ${targetPath})
	#message("outputFilename = " ${targetFilename})
	#message("installPath    = " ${installPath})
	#message("targetVersion   = " ${targetVersion})
	#message("targetSOVersion = " ${targetSOVersion})

	# Ensure, that either both of the version target properties VERSION
	# and SOVERSION is used or none.
	if ((NOT ${targetSOVersion} MATCHES NOTFOUND) AND (${targetVersion} MATCHES NOTFOUND))
		message(FATAL_ERROR "The target " ${target} " uses only one property SOVERSION or VERSION! Always use both or none.")
	endif()


	if(WIN32)
	
		MIRA_DIST_FILE_EX(${target} ${targetPath} ${installPath})
		
		# If the target has a LOCATION, then target is a real build target.
		# In this case we also have to copy the lib file. Otherwise the
		# target is an external or other target. Then we don't have to
		# copy the lib file.
		get_target_property(targetLoc ${target} LOCATION_${CMAKE_BUILD_TYPE})
		if (NOT ${targetLoc} MATCHES NOTFOUND)
			MIRA_DIST_FILE_EX(${target} ${targetPathLib} ${installPathLib})
		endif()

	else(WIN32) # LINUX

		if ((${targetVersion} MATCHES NOTFOUND) AND (${targetSOVersion} MATCHES NOTFOUND))
			# Both properties VERSION and SOVERSION are not used.
			# Only create a symbolic link for the target	
			MIRA_DIST_FILE_EX(${target} ${targetPath} ${installPath})
		else()

			# Both properties VERSION and SOVERSION are used. So create
			# a symbolic links for all three types.
			MIRA_DIST_FILE_EX(${target} ${targetPath} ${installPath}.${targetVersion})
			MIRA_DIST_FILE_EX(${target} ${installPath}.${targetVersion} ${installPath}.${targetSOVersion})
			MIRA_DIST_FILE_EX(${target} ${installPath}.${targetSOVersion} ${installPath})
			
		endif()

	endif(WIN32)

endmacro(MIRA_DIST_LIBRARY_EX)

###############################################################################

macro(MIRA_DIST_LIBRARY target)
	set(subdir ${ARGV1}) # optional 2nd parameter that specifies the subdirectory

	# get the full path to the generated library
	get_target_property(targetPath ${target} LOCATION_${CMAKE_BUILD_TYPE})

	MIRA_DIST_LIBRARY_EX(${target} ${targetPath} ${subdir})

endmacro(MIRA_DIST_LIBRARY)

###############################################################################

# the allowed subdirs
set(ALLOWED_BIN_SUBDIRS tools)

# the root bin dir
set(BIN_DIST_ROOT_DIR ${CMAKE_SOURCE_DIR}/bin)

macro(MIRA_DIST_BINARY target)
	set(subdir ${ARGV1}) # optional 2nd parameter that specifies the subdirectory

	#check if 'subdir' is in the list of allowed subdirectories (but only if we have a subdir)
	if(${ARGC} GREATER 1)
		list(FIND ALLOWED_BIN_SUBDIRS ${subdir} found)
		if(${found} EQUAL -1)
			message("found = " ${found})
			message(FATAL_ERROR "The following subdirectories are allowed for MIRA_DIST_BINARY only: " "${ALLOWED_BIN_SUBDIRS}")
		endif(${found} EQUAL -1)
	endif(${ARGC} GREATER 1)

	# get the full path to the generated binary
	get_target_property(fullOutputPath ${target} LOCATION_${CMAKE_BUILD_TYPE} )

	# extract the binary filename
	get_filename_component(outputFilename ${fullOutputPath} NAME)

	# build the dir and full path where we should put the symbolic link
	set(installDir  ${BIN_DIST_ROOT_DIR}/${subdir})
	set(installPath ${installDir}/${outputFilename})

	#for debugging:
	#message("subdir = " ${subdir})
	#message("fullOutputPath = " ${fullOutputPath})
	#message("outputFilename = " ${outputFilename})
	#message("installDir     = " ${installDir})
	#message("installPath    = " ${installPath})

	MIRA_DIST_FILE_EX(${target} ${fullOutputPath} ${installPath})
	
endmacro(MIRA_DIST_BINARY)

###############################################################################

# the root bin dir
set(FILES_DIST_ROOT_DIR ${CMAKE_SOURCE_DIR})

macro(MIRA_DIST_FILE target source)
	set(subdir ${ARGV2}) # optional 2nd parameter that specifies the subdirectory

	# get the full path to the source file
	set(fullSourcePath ${CMAKE_CURRENT_SOURCE_DIR}/${source})

	# extract the source filename
	get_filename_component(sourceFilename ${fullSourcePath} NAME)

	# build the dir name and full path name of the destination
	set(installDir  ${FILES_DIST_ROOT_DIR}/${subdir})
	set(installPath ${installDir}/${sourceFilename})

	#for debugging:
	#message("subdir = " ${subdir})
	#message("fullSourcePath = " ${fullSourcePath})
	#message("sourceFilename = " ${sourceFilename})
	#message("installDir     = " ${installDir})
	#message("installPath    = " ${installPath})

	MIRA_DIST_FILE_EX(${target} ${fullSourcePath} ${installPath})

endmacro(MIRA_DIST_FILE)


macro(MIRA_DIST_FILES target)
	###########################################################################
	# Parse the macro arguments

	CMAKE_PARSE_ARGUMENTS(arg
		# options:
		""
		# one-value arguments:
		"SUBDIR"
		# multi-value arguments:
		"FILES"
		${ARGN})

	# Ensure, that we have at least one source file
	if (NOT arg_FILES)
		message(FATAL_ERROR "Target ${target} has no files to install.")
	endif()

	if (NOT arg_SUBDIR)
		foreach(item ${arg_FILES})
			MIRA_DIST_FILE(${target} ${item})
		endforeach(item)
	else()
		foreach(item ${arg_FILES})
			MIRA_DIST_FILE(${target} ${item} ${arg_SUBDIR})
		endforeach(item)
	endif()
endmacro(MIRA_DIST_FILES)

###############################################################################
