/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyEditorCommonDelegates.h
 *    Implementation of PropertyEditorCommonDelegates.
 *
 * @author Erik Einhorn
 * @date   2010/12/14
 */

#ifndef _MIRA_PROPERTYEDITORCOMMONDELEGATES_H_
#define _MIRA_PROPERTYEDITORCOMMONDELEGATES_H_

#include <QSlider>
#include <QSpinBox>
#include <QComboBox>
#include <QDoubleSpinBox>
#include <QLineEdit>
#include <QCheckBox>

#include <math/Saturate.h>
#include <serialization/PropertySerializer.h>

#include <widgets/GuiWidgetsExports.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class PropertyNode;

class PropertyDelegate_Slider : public QSlider
{
	Q_OBJECT
public:
	PropertyDelegate_Slider(PropertyNode* p, QWidget* parent);
public slots:
	void slotSetValue(int value);
public:
	PropertyNode* property;
};

///////////////////////////////////////////////////////////////////////////////

class SpinBoxSmartScroll
{
public:
	SpinBoxSmartScroll();
	void handleEventFilter(QEvent* event);
	double handleWheelEvent(QWheelEvent* event, QLineEdit* lineEdit, QChar decimalPoint, double defaultStep);
	void highlightScrolledDigit(QLineEdit* lineEdit, QChar decimalPoint);
	int scrolledDigit() const;
private:
	bool mMouseMovedSinceLastScroll;
	// <0 for digits behind decimal dot, >=0 for digits in front of decimal point, ==0, if no scrolled digit
	int mScrolledDigit;
	double mToAdd;
};

///////////////////////////////////////////////////////////////////////////////

// helper functions, overloads allow selecting matching QString conversion function
bool stringToLL(const QString& input, int64& val);
bool stringToLL(const QString& input, uint64& val);

/**
 * A spinbox for integral values of any type T (intended to be used with 64bit values),
 * implemented by subclassing QAbstractSpinBox.
 */
template<typename T>
class LongSpinBoxBase : public QAbstractSpinBox
{
public:
	explicit LongSpinBoxBase(QWidget* parent = 0) : QAbstractSpinBox(parent) {
		lineEdit()->setText(textFromValue(m_value));
	}

	T value() const
	{
		return m_value;
	}

	T minimum() const
	{
		return m_minimum;
	}

	void setMinimum(T min)
	{
		m_minimum = min;
		m_maximum = std::max(min, m_maximum);
	}

	T maximum() const
	{
		return m_maximum;
	}

	void setMaximum(T max)
	{
		m_maximum = max;
		m_minimum = std::min(max, m_minimum);
	}

	void setRange(T min, T max)
	{
		setMinimum(min);
		setMaximum(max);
	}

	StepEnabled stepEnabled() const
	{
		return StepUpEnabled | StepDownEnabled;
	}

	T singleStep()
	{
		return m_step;
	}

	void setSingleStep(T step)
	{
		m_step = std::max<T>(0, step);
	}

	virtual void stepBy(int steps)
	{
		if (steps == 0)
			return;

		auto new_value = m_value + steps * m_step;
		if (steps < 0 && new_value > m_value) {
			new_value = std::numeric_limits<T>::min();
		} else if (steps > 0 && new_value < m_value) {
			new_value = std::numeric_limits<T>::max();
		}

		new_value = saturate(new_value, m_minimum, m_maximum);

		lineEdit()->setText(textFromValue(new_value));
		setValue(new_value);
	}

protected:
	virtual QValidator::State validate(QString& input, int& pos) const
	{
		T val;
		bool ok = stringToLL(input, val);
		if (!ok)
			return QValidator::Invalid;

		if (val < m_minimum || val > m_maximum)
			return QValidator::Invalid;

		return QValidator::Acceptable;
	}

	virtual T valueFromText(const QString& text) const
	{
		T val;
		stringToLL(text, val);
		return val;
	}

	virtual QString textFromValue(T val) const
	{
		return QString::number(val);
	}

public:
	virtual void setValue(T val)
	{
		if (m_value == val)
			return;

		m_value = val;
		lineEdit()->setText(textFromValue(val));
	}

	void onEditFinished()
	{
		QString input = lineEdit()->text();
		int pos;
		if (QValidator::Acceptable == validate(input, pos))
			setValue(valueFromText(input)); // this e.g. normalizes (valid) input like '1   '
		else
			lineEdit()->setText(textFromValue(m_value));
	}

protected:
	T m_minimum;
	T m_maximum;
	T m_step;
	T m_value;
};

class MIRA_GUI_WIDGETS_EXPORT PropertyDelegate_SpinBoxSigned
	: public LongSpinBoxBase<int64>, private SpinBoxSmartScroll
{
	typedef LongSpinBoxBase<int64> Base;

	// Q_OBJECT cannot be used in the base (template) class,
	// so Qt properties/signals/slots are defined in this specialization
	Q_OBJECT

	Q_PROPERTY(int64 minimum READ minimum WRITE setMinimum)
	Q_PROPERTY(int64 maximum READ maximum WRITE setMaximum)

	Q_PROPERTY(int64 value READ value WRITE setValue NOTIFY valueChanged USER true)

public:
	PropertyDelegate_SpinBoxSigned(PropertyNode* p, QWidget* parent);

	/// call Base::setValue and emit valueChanged(val)
	void setValue(int64 val) override;

	/// convenience function, sets limits matching a type's numeric limits (but PropertyHint takes priority!)
	template<typename T>
	void limitRangeToType()
	{
		static_assert(std::is_integral<T>::value, "PropertyDelegate_SpinBoxSigned::limitTypeRange() requires integral type.");
		static_assert(std::is_signed<T>::value, "PropertyDelegate_SpinBoxSigned::limitTypeRange() requires signed type.");

		setMinimum(property->getHint<int64>("minimum", std::numeric_limits<T>::min()));
		setMaximum(property->getHint<int64>("maximum", std::numeric_limits<T>::max()));
	}

public slots:
	void onEditFinished();
	void slotSetValue(int64 value);

signals:
	void valueChanged(int64 v);

public:
	PropertyNode* property;
private:
	void wheelEvent(QWheelEvent* event);
protected:
	bool eventFilter(QObject* obj, QEvent* event);
};

class MIRA_GUI_WIDGETS_EXPORT PropertyDelegate_SpinBoxUnsigned
	: public LongSpinBoxBase<uint64>, private SpinBoxSmartScroll
{
	typedef LongSpinBoxBase<uint64> Base;

	// Q_OBJECT cannot be used in the base (template) class,
	// so Qt properties/signals/slots are defined in this specialization
	Q_OBJECT

	Q_PROPERTY(uint64 minimum READ minimum WRITE setMinimum)
	Q_PROPERTY(uint64 maximum READ maximum WRITE setMaximum)

	Q_PROPERTY(uint64 value READ value WRITE setValue NOTIFY valueChanged USER true)

public:
	PropertyDelegate_SpinBoxUnsigned(PropertyNode* p, QWidget* parent);

	/// call Base::setValue and emit valueChanged(val)
	void setValue(uint64 val);

	/// convenience function, sets limits matching a type's numeric limits (but PropertyHint takes priority!)
	template<typename T>
	void limitRangeToType()
	{
		static_assert(std::is_integral<T>::value, "PropertyDelegate_SpinBoxUnsigned::limitTypeRange() requires integral type.");
		static_assert(!std::is_signed<T>::value, "PropertyDelegate_SpinBoxUnsigned::limitTypeRange() requires unsigned type.");

		setMinimum(property->getHint<uint64>("minimum", std::numeric_limits<T>::min()));
		setMaximum(property->getHint<uint64>("maximum", std::numeric_limits<T>::max()));
	}

public slots:
	void onEditFinished();
	void slotSetValue(uint64 value);

signals:
	void valueChanged(uint64 v);

public:
	PropertyNode* property;
private:
	void wheelEvent(QWheelEvent* event);
protected:
	bool eventFilter(QObject* obj, QEvent* event);
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_DoubleSlider : public QSlider
{
	Q_OBJECT
public:
	PropertyDelegate_DoubleSlider(PropertyNode* p, QWidget* parent);
public slots:
	void slotSetValue(int value);
public:
	PropertyNode* property;
private:
	double mMinimum;
	double mMaximum;
	double mStep;
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_DoubleSpinBox : public QDoubleSpinBox, private SpinBoxSmartScroll
{
	Q_OBJECT
public:
	PropertyDelegate_DoubleSpinBox(PropertyNode* p, QWidget* parent);
	QString textFromValue(double value) const;
	double valueFromText(const QString& text) const;
public slots:
	void slotSetValue(double value);
public:
	PropertyNode* property;
	int precision;
	QLocale locale;
private:
	void wheelEvent(QWheelEvent* event);
protected:
	bool eventFilter(QObject* obj, QEvent* event);
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_ComboBox : public QComboBox
{
	Q_OBJECT
public:
	PropertyDelegate_ComboBox(PropertyNode* p, QWidget* parent);
public slots:
	void slotSetValue(int value);
public:
	PropertyNode* property;
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_LineEdit : public QLineEdit
{
	Q_OBJECT
public:
	PropertyDelegate_LineEdit(PropertyNode* p, QWidget* parent);
public slots:
	void slotSetValue();
public:
	PropertyNode* property;
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_CheckBox : public QCheckBox
{
	Q_OBJECT
public:
	PropertyDelegate_CheckBox(PropertyNode* p, QWidget* parent);
public slots:
	void slotSetValue(bool value);
public:
	TypedPropertyNode<bool>* property;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
