/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationRegistry.C
 *    Implementation of VisualizationRegistry.
 *
 * @author Erik Einhorn
 * @date   2011/01/02
 */

#include <error/LoggingCore.h>

#include <visualization/Visualization.h>
#include <visualization/VisualizationRegistry.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

std::list<ClassProxy> VisualizationRegistry::getVisualizationsFor(
		const Typename& type, const Class& visualizationType)
{
	// check if we have new visualizations
	lookForNewVisualizations();

	std::list<ClassProxy> res;
	auto r = mKnownVisualizations.equal_range(type);

	for(auto it=r.first; it!=r.second; ++it)
	{
		ClassProxy c = it->second.visClass;
		if( ( visualizationType.isBaseOf(c) ||
					visualizationType.getIdentifier() == c.getIdentifier() )
				&& !c.isAbstract() ) { // class c is of the desired type
			res.push_back(c);
		}
	}

	return res;
}

void VisualizationRegistry::lookForNewVisualizations()
{
	typedef std::map<std::string, ClassProxy > ClassMap;
	ClassMap classes = Visualization::CLASS().getDerivedClasses();

	if(classes.size()<=mKnownVisualizationsClasses.size())
		return; // already abort here, if there are no new classes since last time

	// loop through all classes
	foreach(ClassMap::value_type& c, classes)
	{

		ClassProxy clazz = c.second;
		if(mKnownVisualizationsClasses.count(clazz)==0) { // if this class is new

			mKnownVisualizationsClasses.insert(clazz); // store it

			if(clazz.isAbstract())
				continue;

			MIRA_LOG(DEBUG) << "Found new visualization: " << c.first;

			// instantiate the visualization to get all information about it
			try {

				Visualization* vis = clazz.newInstance<Visualization>();

				Visualization::DataConnection connection = vis->getDataConnection();

				if(!connection.type.empty() && connection.property!=NULL) {
					MIRA_LOG(DEBUG) << "... visualization is useful for type '" << connection.type << "'";

					// add the class with all information
					mKnownVisualizations.insert(
							std::make_pair(connection.type, Entry(connection.type, clazz)));
				}

				// we are done with the visualization, so destroy it
				delete vis;

			} catch(Exception& ex) {
				MIRA_LOG(ERROR) << "Failed to load visualization '" << c.first << "': " << ex.message();
			}

		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
