/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file EditorPartArea.C
 *    Implementation of EditorPartArea.h.
 *
 * @author Erik Einhorn
 * @date   2011/09/11
 */

#include <math.h>

#include <utils/Foreach.h>
#include <rcp/EditorPartArea.h>

#include <assert.h>

#include <iostream>

#include <QEvent>
#include <QCoreApplication>

#include <QAbstractScrollArea>
#include <QApplication>
#include <QPainter>
#include <QPaintEvent>
#include <QScrollBar>
#include <QStyle>

#include <rcp/EditorPartWindow.h>
#include <rcp/EditorPartTitleBar.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class EditorPartArea::ScrollArea : public QAbstractScrollArea
{
public:

	ScrollArea(EditorPartArea* parent) : QAbstractScrollArea(parent),
		mArea(parent)
	{
		setBackground(palette().brush(QPalette::Dark));
		setFrameStyle(QFrame::NoFrame);
		setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
		setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);

		setViewport(NULL);
		setFocusPolicy(Qt::NoFocus);
	}

	void setBackground(const QBrush &brush)
	{
		if(mBackground != brush) {
			mBackground = brush;
			viewport()->setAttribute(Qt::WA_OpaquePaintEvent, brush.isOpaque());
			update();
		}
	}

protected:

	void paintEvent(QPaintEvent *paintEvent)
	{
		QPainter painter(viewport());
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
		foreach (const auto& r, paintEvent->region())
			painter.fillRect(r, mBackground);
#else
		const QVector<QRect>& exposedRects = paintEvent->region().rects();
		for (int i = 0; i < exposedRects.size(); ++i)
			painter.fillRect(exposedRects.at(i), mBackground);
#endif
	}


	virtual void scrollContentsBy(int dx, int dy)
	{
		viewport()->scroll(dx, dy);
		mArea->scrolledAreaBy(dx, dy);
	}

	virtual void resizeEvent(QResizeEvent* event) {
		updateScrollBars();
		mArea->resizeMaximizedSubwindows(event->size());
	}

public:

	void updateScrollBars()
	{

		QSize areaSize = viewport()->size();
		const QRect childrenRect = viewport()->childrenRect();

		int xoff = childrenRect.left() + horizontalScrollBar()->value();
		int yoff = childrenRect.top() + verticalScrollBar()->value();
		horizontalScrollBar()->setRange(std::min(0,xoff),
		                                std::max(0,xoff + childrenRect.width() - areaSize.width()));
		horizontalScrollBar()->setPageStep(childrenRect.width());
		horizontalScrollBar()->setSingleStep(childrenRect.width() / 20);

		verticalScrollBar()->setRange(std::min(0,yoff),
		                              std::max(0,yoff + childrenRect.height() - areaSize.height()));
		verticalScrollBar()->setPageStep(childrenRect.height());
		verticalScrollBar()->setSingleStep(childrenRect.height() / 20);
	}

	// similar to contentsRect but takes scroll bars into account that reduce the size
	QRect actualContentsRect() const
	{
		return viewport()->contentsRect();
	}

private:

	QBrush mBackground;
	EditorPartArea* mArea;

};

///////////////////////////////////////////////////////////////////////////////

EditorPartArea::EditorPartArea(QWidget* parent, Qt::WindowFlags flags) :
		QWidget(parent)
{
	mAutoHideDecorations = false;

	mArea = new ScrollArea(this);
	mTabBar = new EditorPartAreaTabBar(this);
	mTabBar->setStyleSheet("QTabBar::tab { height: 24px; min-width: 120px; }");

#if (QT_VERSION >= QT_VERSION_CHECK(4, 5, 0))
	mTabBar->setMovable(true);
#endif

	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setSpacing(0);
	layout->setMargin(0);

	QHBoxLayout* tablayout = new QHBoxLayout();
	tablayout->setSpacing(0);
	tablayout->setMargin(0);
	tablayout->addWidget(mTabBar);
	tablayout->addStretch();

	layout->addLayout(tablayout);
	layout->addWidget(mArea);

	setFocusPolicy(Qt::NoFocus);
	mActiveSubWindow = NULL;

	connect(mTabBar, SIGNAL(currentChanged(int)), this, SLOT(onTabChanged(int)));
	connect(mTabBar, SIGNAL(tabDoubleClicked(int)), this, SLOT(onTabDoubleClicked(int)));

	connect(qApp, SIGNAL(focusChanged(QWidget*,QWidget*)),
	        this, SLOT(onFocusChanged(QWidget*,QWidget*)));
}

void EditorPartArea::addSubWindow(EditorPartWindow* window)
{
	if(mSubWindows.count(window)>0)
		return; // already added

	assert(window->mArea==NULL);

	window->mArea = this;

	// reparent
	window->setParent(mArea->viewport());
	window->show();
	window->installEventFilter(this);
	window->setAutoHideDecorations(mAutoHideDecorations);

	mSubWindows.insert(window);
	mDockedSubWindows.insert(window); // at the beginning the window is docked

	if(activeSubWindow()==NULL)
		activateSubWindow(window);

	updateMaximizedSubwindowsVisibility();
}

void EditorPartArea::removeSubWindow(EditorPartWindow* window)
{
	if(mSubWindows.count(window)==0)
		return; // not added

	window->setParent(NULL);
	window->removeEventFilter(this);
	mSubWindows.erase(window);
	mDockedSubWindows.erase(window);

	removeWindowTab(window); // remove tab if any

	if(mActiveSubWindow==window)
		mActiveSubWindow=NULL;

	updateMaximizedSubwindowsVisibility();
}

void EditorPartArea::activateSubWindow(EditorPartWindow* window)
{
	if(mSubWindows.count(window)==0)
		return; // not added

	if(activeSubWindow()!=window)
	{
		if(activeSubWindow())
			activeSubWindow()->setActive(false);

		mActiveSubWindow=window;
		emit editorWindowActivated(window);
	}

	window->setActive(true);

	int idx = findWindowTab(window);
	if(idx>=0 && mTabBar->currentIndex()!=idx)
		mTabBar->setCurrentIndex(idx);

	internalRaise(window);

	QWidget* focusWidget = QApplication::focusWidget();

	// set focus, if we or our children do not own the focus already
	if(!(focusWidget == window) && !window->isAncestorOf(focusWidget))
		window->setFocus(Qt::ActiveWindowFocusReason);

	updateMaximizedSubwindowsVisibility();
}

EditorPartWindow* EditorPartArea::activeSubWindow () const
{
	return mActiveSubWindow;
}

void EditorPartArea::scrollToSubWindow(EditorPartWindow* window)
{
	QRect wgeo = window->geometry();
	QScrollBar* hor = mArea->horizontalScrollBar();
	QScrollBar* ver = mArea->verticalScrollBar();

	// if the window does not fit in the area, scroll to center its center
	// otherwise scroll to make window's edges visible, if necessary

	if (wgeo.width() > geometry().width()) {
		hor->setValue(hor->value() + wgeo.left() + (wgeo.width() - geometry().width())/2);
	} else {
		if (wgeo.left() < 0)
			hor->setValue(hor->value() + wgeo.left());

		if (wgeo.right() > geometry().width())
			hor->setValue(hor->value() + wgeo.right() - geometry().width());
	}

	if (wgeo.height() > geometry().height()) {
		ver->setValue(ver->value() + wgeo.top() + (wgeo.height() - geometry().height())/2);
	} else {
		if (wgeo.top() < 0)
			ver->setValue(ver->value() + wgeo.top());

		if (wgeo.bottom() > geometry().height())
			ver->setValue(ver->value() + wgeo.bottom() - geometry().height());
	}
}

void EditorPartArea::tileSubWindows()
{
	// collect the docked and non maximized windows for tiling
	std::vector<EditorPartWindow*> widgets;

	QObjectList children = mArea->viewport()->children();
	foreach(QObject *object, children)
	{
		EditorPartWindow *child = qobject_cast<EditorPartWindow*>(object);
		if (!child || !mDockedSubWindows.count(child) || child->isMaximized())
			continue;
		widgets.push_back(child);
	}


	QRect domain = mArea->viewport()->rect();

	// arrange the widgets
	if(widgets.empty())
		return;


	// Tiling algorithm was taken from Qt 4.7: qmdiarea.cpp, provided under
	// GNU Lesser General Public License
	const int n = widgets.size();
	const int ncols = std::max((int)ceil(sqrt((float)n)), 1);
	const int nrows = std::max((n % ncols) ? (n / ncols + 1) : (n / ncols), 1);
	const int nspecial = (n % ncols) ? (ncols - n % ncols) : 0;
	const int dx = domain.width()  / ncols;
	const int dy = domain.height() / nrows;

	int i = 0;
	for (int row = 0; row < nrows; ++row) {
		const int y1 = int(row * (dy + 1));
		for (int col = 0; col < ncols; ++col) {
			if (row == 1 && col < nspecial)
				continue;
			const int x1 = int(col * (dx + 1));
			int x2 = int(x1 + dx);
			int y2 = int(y1 + dy);
			if (row == 0 && col < nspecial) {
				y2 *= 2;
				if (nrows != 2)
					y2 += 1;
				else
					y2 = domain.bottom();
			}
			if (col == ncols - 1 && x2 != domain.right())
				x2 = domain.right();
			if (row == nrows - 1 && y2 != domain.bottom())
				y2 = domain.bottom();
			QWidget *widget = widgets[i++];
			QRect newGeometry = QRect(QPoint(x1, y1), QPoint(x2, y2));
			widget->setGeometry(QStyle::visualRect(widget->layoutDirection(), domain, newGeometry));
		}
	}
}

void EditorPartArea::setBackground(const QBrush& brush)
{
	assert(mArea);
	mArea->setBackground(brush);
}

void EditorPartArea::setAutoHideDecorations(bool on)
{
	mAutoHideDecorations = on;

	foreach(EditorPartWindow* w, mSubWindows)
		w->setAutoHideDecorations(on);
}

void EditorPartArea::dockSubwindow(EditorPartWindow* window)
{
	assert(mSubWindows.count(window)==1);

	if(mDockedSubWindows.count(window)==1)
		return; // already docked

	mDockedSubWindows.insert(window);

	bool maximized = window->isMaximized();
	QPoint p = mArea->viewport()->mapFromGlobal(window->pos());
	if(p.x() < 0) p.setX(0);
	if(p.y() < 0) p.setY(0);

	int width  = this->width()-50;
	int height = this->height()-50;
	if(p.x() > width)  p.setX(width);
	if(p.y() > height) p.setY(height);

	window->setParent(mArea->viewport());
	window->move(p);

	if(maximized) {
		window->showMaximized();
		rearrangeMaximizedSubwindow(window);
		addWindowTab(window);
	} else
		window->showNormal();

	activateSubWindow(window);
}

void EditorPartArea::undockSubwindow(EditorPartWindow* window)
{
	assert(mSubWindows.count(window)==1);

	if(mDockedSubWindows.count(window)==0)
		return; // not docked

	mDockedSubWindows.erase(window);

	bool maximized = window->isMaximized();
	QPoint p = mArea->viewport()->mapToGlobal(window->pos());

	window->setParent(NULL);
	window->move(p);

	if(maximized) {
//		window->showMaximized(); // this would be nice probably, but is not
		                         // guaranteed not to screw with our windows
		                         // https://mira-project.org/trac/ticket/819
		window->showNormal();
		removeWindowTab(window);
	} else
		window->showNormal();

	updateMaximizedSubwindowsVisibility();
}

void EditorPartArea::maximizedSubwindow(EditorPartWindow* window)
{
	if(mDockedSubWindows.count(window)!=1)
		return;
	rearrangeMaximizedSubwindow(window);

	addWindowTab(window);
	activateSubWindow(window);
}

void EditorPartArea::restoredSubwindow(EditorPartWindow* window)
{
	if(mDockedSubWindows.count(window)!=1)
		return;

	removeWindowTab(window);
	activateSubWindow(window);
}

void EditorPartArea::rearrangeMaximizedSubwindow(EditorPartWindow* window)
{

	if(mDockedSubWindows.count(window)!=1)
		return;

	QRect availableRect = mArea->actualContentsRect();

	if(window->geometry() == availableRect)
		return;

	// hide window before changing the geometry to avoid multiple resize
	// events and wrong window states
	bool wasVisible = window->isVisible();
	if(wasVisible)
		window->setVisible(false);

	window->setGeometry(availableRect);

	// QWidget::setGeometry will reset Qt::WindowMaximized so reset it here
	window->setWindowState(Qt::WindowMaximized);

	if (wasVisible)
		window->setVisible(true);
}

void EditorPartArea::rearrangeMaximizedSubwindows()
{
	foreach (EditorPartWindow* child, mDockedSubWindows) {
		if (child->isMaximized())
			rearrangeMaximizedSubwindow(child);
	}
}

/**
 * Hides all maximized windows that are below the toplevel maximized window.
 * The topmost maximized window is set visible. (the topmost window is
 * determined using the tab which is currently active)
 */
void EditorPartArea::updateMaximizedSubwindowsVisibility()
{
	int currentTab = mTabBar->currentIndex();

	foreach (EditorPartWindow* child, mDockedSubWindows) {
		if (child->isMaximized()) {
			int idx = findWindowTab(child);
			child->setVisible(idx==currentTab);
		}

	}
}

void EditorPartArea::resizeMaximizedSubwindows(const QSize& size)
{
	foreach (EditorPartWindow* child, mDockedSubWindows)
	{
		if (child->isMaximized()) {
			child->resize(size);
			// QWidget::setGeometry will reset Qt::WindowMaximized so reset it here
			child->setWindowState(Qt::WindowMaximized);
			child->blockSignals(false);
		}
	}
}

void EditorPartArea::scrolledAreaBy(int dx, int dy)
{
	rearrangeMaximizedSubwindows();
}

std::set<EditorPartWindow*> EditorPartArea::getSubWindows() {
	return mSubWindows;
}

void EditorPartArea::onFocusChanged(QWidget *old, QWidget *now)
{
	foreach(EditorPartWindow* w, mSubWindows)
	{
		if (now && (now == w || w->isAncestorOf(now))) {
			activateSubWindow(w);
			return;
		}
	}
}

bool EditorPartArea::eventFilter(QObject *object, QEvent *event)
{
	EditorPartWindow* window = dynamic_cast<EditorPartWindow*>(object);
	if (window==NULL)
		return QWidget::eventFilter(object, event);

	switch(event->type())
	{
	case QEvent::Move:
	case QEvent::Resize:
		mArea->updateScrollBars();
		rearrangeMaximizedSubwindows();
		break;
	default:
		break;
	}

	return QWidget::eventFilter(object, event);
}

void EditorPartArea::resizeEvent(QResizeEvent* event)
{
	// resize all maximized (docked) subwindows
	//rearrangeMaximizedSubwindows();
	//mArea->updateScrollBars();
}

void EditorPartArea::onTabChanged(int idx)
{
	QObject* o = mTabBar->tabData(idx).value<QObject*>();
	if(o!=NULL)
		activateSubWindow((EditorPartWindow*)o);
}

void EditorPartArea::onTabDoubleClicked(int idx)
{
	QObject* o = mTabBar->tabData(idx).value<QObject*>();
	if(o!=NULL) {
		EditorPartWindow* w = (EditorPartWindow*)o;
		w->showNormal();
	}
}

void EditorPartArea::addWindowTab(EditorPartWindow* window)
{
	if(findWindowTab(window)>=0)
		return; // already added

	mTabBar->blockSignals(true);
	int idx = mTabBar->addTab(window->windowTitle());
	mTabBar->setTabData(idx, QVariant::fromValue( (QObject*)window ) );
	mTabBar->setCurrentIndex(idx);
	mTabBar->blockSignals(false);

	mTabBar->setTabButton(idx, QTabBar::LeftSide,  window->mTitleBar->takeLeftButtons());
	mTabBar->setTabButton(idx, QTabBar::RightSide, window->mTitleBar->takeRightButtons());
}

void EditorPartArea::removeWindowTab(EditorPartWindow* window)
{
	int idx = findWindowTab(window);
	if(idx<0)
		return;
	
	mTabBar->setTabButton(idx, QTabBar::LeftSide, NULL);
	mTabBar->setTabButton(idx, QTabBar::RightSide, NULL);
	window->mTitleBar->putBackLeftButtons();
	window->mTitleBar->putBackRightButtons();

	mTabBar->blockSignals(true);
	mTabBar->removeTab(idx);
	mTabBar->blockSignals(false);
}

int EditorPartArea::findWindowTab(EditorPartWindow* window) const
{
	if(window==NULL)
		return -1;

	QObject* w = window;

	int tabCount = mTabBar->count();
	for(int i=0; i<tabCount;++i)
	{
		QObject* wi = mTabBar->tabData(i).value<QObject*>();
		if(wi==w)
			return i;
	}
	return -1;
}

std::vector<EditorPartWindow*> EditorPartArea::getTabbedWindows()
{
	std::vector<EditorPartWindow*> windows;

	int tabCount = mTabBar->count();
	for(int i=0; i<tabCount;++i) {
		// https://doc.qt.io/qt-5/qvariant.html#value
		EditorPartWindow* wi = mTabBar->tabData(i).value<EditorPartWindow*>();
		if (wi)
			windows.push_back(wi);
	}

	return windows;
}

void EditorPartArea::internalRaise(EditorPartWindow *window) const
{
	if(mDockedSubWindows.size() < 2)
		return;

	EditorPartWindow *stackUnderChild = NULL;
	QObjectList children = mArea->viewport()->children();
	foreach_reverse (QObject *object, children)
	{
		EditorPartWindow *child = qobject_cast<EditorPartWindow*>(object);
		if (!child || !mDockedSubWindows.count(child))
			continue;

		if ( (window->isMaximized() || (child->isOnTop() && !window->isOnTop())) &&
			 !child->isHidden() && !child->isMaximized())
		{
			if (stackUnderChild)
				child->stackUnder(stackUnderChild);
			else
				child->raise();
			stackUnderChild = child;
		}
	}

	if (stackUnderChild)
		window->stackUnder(stackUnderChild);
	else
		window->raise();

}

///////////////////////////////////////////////////////////////////////////////

}
