/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeTapeRecorderWidget.C
 *    Implementation of TapeRecorderWidget.h.
 *
 * @author Tim Langner
 * @date   2012/02/03
 */

#include <fw/TapeRecorderWidget.h>

#include <boost/format.hpp>

#include <QMimeData>
#include <QUrl>

#include <QComboBox>
#include <QDragEnterEvent>
#include <QFileDialog>
#include <QHBoxLayout>
#include <QToolButton>
#include <QSpinBox>
#include <QVBoxLayout>
#include <QHeaderView>
#include <QProgressDialog>

#include <fw/Framework.h>

#include <views/GUIMimeTypes.h>
#include <widgets/SelectionListDialog.h>
#include <widgets/QtUtils.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// A combo box that shows long text in popup and short text when popup is hidden 
class ComboBox : public QComboBox
{
public:
	ComboBox(QWidget *parent,
	         const QStringList& iLong, const QStringList& iShort)
		: QComboBox(parent), mLong(iLong), mShort(iShort)
		{
			assert(mShort.size() == mLong.size());
			addItems(mShort);
		}

public:
	virtual void showPopup()
	{
		for (int i = 0; i < mShort.size(); ++i)
			setItemText(i, mLong.at(i));
		QComboBox::showPopup();
	}

	virtual void hidePopup()
	{
		for (int i = 0; i < mLong.size(); ++i)
			setItemText(i, mShort.at(i));
		QComboBox::hidePopup();
	}

private:
	QStringList mLong, mShort;	
};

///////////////////////////////////////////////////////////////////////////////

CodecDialog::CodecDialog(QWidget* parent) : QDialog(parent)
{
	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);
	QPushButton* addRemoveBtn = new QPushButton("Add/Remove", this);
	connect(addRemoveBtn, SIGNAL(clicked()), this, SLOT(addRemove()));
	layout->addWidget(addRemoveBtn);
	mEditor = new PropertyEditor(this);
	layout->addWidget(mEditor);
	mButtonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel);

	connect(mButtonBox,  SIGNAL(accepted()), this, SLOT(accept()));
	connect(mButtonBox,  SIGNAL(rejected()), this, SLOT(reject()));
	layout->addWidget(mButtonBox);
}

CodecDialog::~CodecDialog()
{
	// need to remove all properties from the editor
	// before deleting them, as the editor will try to unregister
	// from the PropertyNodes in its destructor!
	mEditor->clear();
}

void CodecDialog::setCodecs(std::list<BinarySerializerCodecPtr>& codecs)
{
	mEditor->clear();
	foreach(auto ptr, codecs)
		addCodec(ptr);
}

std::list<BinarySerializerCodecPtr> CodecDialog::getCodecs()
{
	std::list<BinarySerializerCodecPtr> codecs;
	foreach(auto c, mCodecs)
		codecs.push_back(c.second.first);
	return codecs;
}

void CodecDialog::addCodec(BinarySerializerCodecPtr codec)
{
	PropertySerializer s;
	PropertyNode* p = s.reflectProperties(codec->getClass().getName(), codec);
	auto pair = std::make_pair(codec, boost::shared_ptr<PropertyNode>(p));
	mCodecs[codec->getClass().getIdentifier()] = pair;
	mEditor->addProperty(p);
}

void CodecDialog::addRemove()
{
	SelectionListDialog<ClassProxy> dialog(this);
	dialog.setWindowTitle(tr("Select/deselect codecs"));
	dialog.setHeaderLabels(QStringList() << "Codec" << "Description");
	dialog.setMultiSelection(true);
	dialog.setSelectOnlyOneItemPerCategory(true);
	dialog.resize(600,400);

	// query all codec classes from the class factory:
	typedef std::map<std::string, ClassProxy > ClassMap;
	ClassMap codecClasses = BinarySerializerCodec::CLASS().getDerivedClasses( );

	foreach(ClassMap::value_type i, codecClasses)
	{
		if(i.second.isAbstract())
			continue;

		std::string name = i.second.getMetaInfo("Name");
		if(name.empty())
			name = i.first; // use class name, if no meta name was specified

		std::string category = i.second.getMetaInfo("Category");
		if(category.empty())
			category = "General"; // use default category

		bool selected = (mCodecs.count(i.second.getIdentifier())>0);

		QStringList labels;
		labels << name.c_str() << i.second.getMetaInfo("Description").c_str();
		dialog.addItem(category.c_str(), labels, i.second, selected);
	}

	if (dialog.exec() == 0)
		return;
	auto selection = dialog.selectedItems();
	// check for items that need to be removed
	typedef std::list<std::pair<QString, std::string>> SelectionType;
	for(auto it = mCodecs.begin(); it != mCodecs.end();)
	{
		bool found = false;
		foreach(const auto& s, selection)
			if (it->first == s.second.getIdentifier())
			{
				found = true;
				break;
			}
		if (!found)
		{
			auto next = it;
			++next;
			mEditor->removeProperty(it->second.second.get());
			mCodecs.erase(it);
			it = next;
		}
		else
			++it;
	}
	// check for new and existing items
	foreach(const auto& s, selection)
	{
		// already in the list continue
		if (mCodecs.count(s.second.getIdentifier())>0)
			continue;
		// add new codec
		BinarySerializerCodecPtr ptr(s.second.newInstance<BinarySerializerCodec>());
		addCodec(ptr);
	}
}

//////////////////////////////////////////////////////////////////////////////

CodecWidget::CodecWidget(QWidget* parent) : QWidget(parent)
{
	QHBoxLayout* layout = new QHBoxLayout(this);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 0);
	mTextLbl = new QLabel("", this);
	mTextLbl->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
	layout->addWidget(mTextLbl);
	QToolButton* selectBtn = new QToolButton(this);
	selectBtn->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Ignored);
	selectBtn->setFixedWidth(20);
	selectBtn->setText(tr("..."));
	connect(selectBtn, SIGNAL(clicked()), this, SLOT(selectCodecs()));
	layout->addWidget(selectBtn, Qt::AlignRight);
}

void CodecWidget::selectCodecs()
{
	CodecDialog dialog(this);
	dialog.setCodecs(mCodecs);
	if (dialog.exec() == 0)
		return;
	setCodecs(dialog.getCodecs());
}

void CodecWidget::setCodecs(const std::list<BinarySerializerCodecPtr>& codecs)
{
	mCodecs = codecs;
	QString txt;
	bool first = true;
	foreach(const auto& c, mCodecs)
	{
		if (!first)
			txt += "|";
		else
			first = false;
		txt += c->getClass().getName().c_str();
	}
	mTextLbl->setText(txt);
}

std::list<BinarySerializerCodecPtr> CodecWidget::getCodecs()
{
	return mCodecs;
}

//////////////////////////////////////////////////////////////////////////////

DurationDialog::DurationDialog(QWidget* parent) : QDialog(parent)
{
	QVBoxLayout* layout = new QVBoxLayout(this);
	QLabel* label1 = new QLabel("Select maximum duration to record (hh:mm:ss):", this);
	layout->addWidget(label1);

	QHBoxLayout* layout2 = new QHBoxLayout();
	layout->addLayout(layout2);

	mTimeEdit = new QTimeEdit(this);
	mTimeEdit->setDisplayFormat("hh:mm:ss");
	layout2->addStretch();
	layout2->addWidget(mTimeEdit);
	layout2->addStretch();

	QLabel* label2 = new QLabel("Time starts counting when recording starts (after selecting tape file).\n"
	                            "Recording will stop automatically after the selected duration,\n"
	                            "if not stopped manually earlier.",
	                            this);
	QFont font = label2->font();
	font.setPointSizeF(font.pointSizeF()*0.8);
	label2->setFont(font);
	layout->addWidget(label2);

	mButtonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel);

	connect(mButtonBox,  SIGNAL(accepted()), this, SLOT(accept()));
	connect(mButtonBox,  SIGNAL(rejected()), this, SLOT(reject()));
	layout->addWidget(mButtonBox);
}

void DurationDialog::setDuration(const Duration& duration)
{
	mTimeEdit->setTime(QTime(duration.hours(), duration.minutes(), duration.seconds()));
}

Duration DurationDialog::getDuration()
{
	QTime time = mTimeEdit->time();
	return Duration::seconds(time.hour()*3600 + time.minute()*60+time.second());
}

//////////////////////////////////////////////////////////////////////////////

TapeRecorderWidget::TapeRecorderWidget(QWidget* parent) : QWidget(parent)
{
	QVBoxLayout* vLayout = new QVBoxLayout(this);
	vLayout->setContentsMargins(0, -1, 0, 0);

	QHBoxLayout* hLayout1 = new QHBoxLayout();
	hLayout1->setContentsMargins(0, 0, 0, 0);
	vLayout->addLayout(hLayout1);

	mChannels = new QTableWidget(0, 8, this);
	QStringList labels;
	//         0       1           2           3              4                    5                   6       7
	labels << "" << "Channel" << "Codecs" << "CompressLvl" << "Interval in ms" <<  "Avoid data loss " << "#" << "MB/s";
	mChannels->setHorizontalHeaderLabels(labels);
	mChannels->setMinimumHeight(10);
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
	mChannels->horizontalHeader()->setSectionResizeMode(0, QHeaderView::Fixed);
	mChannels->verticalHeader()->setSectionResizeMode(QHeaderView::Fixed);	
#else
	mChannels->horizontalHeader()->setResizeMode(0, QHeaderView::Fixed);      	
	mChannels->verticalHeader()->setResizeMode(QHeaderView::Fixed);
#endif
	mChannels->horizontalHeader()->resizeSection(0, 24);

	hLayout1->addWidget(mChannels);

	QHBoxLayout* hLayout2 = new QHBoxLayout();
	hLayout2->setContentsMargins(0, 0, 0, 0);

	mBtRecord = new QToolButton(this);
	mBtRecord->setFixedSize(32,32);
	mBtRecord->setAutoRaise(true);
	mBtRecord->setToolTip("Start recording. You will be asked to select a tape file name.");
	mBtRecord->setIcon(QIcon(":/icons/Record.png"));
	mBtRecord->setEnabled(false);
	connect(mBtRecord, SIGNAL(clicked()), this, SLOT(record()) );
	hLayout2->addWidget(mBtRecord);

	mBtRecordDuration = new QToolButton(this);
	mBtRecordDuration->setFixedSize(32,32);
	mBtRecordDuration->setAutoRaise(true);
	mBtRecordDuration->setToolTip("Start recording for a maximum duration. "
	                           "You will be asked to select a duration and a tape file name.");
	mBtRecordDuration->setIcon(QIcon(":/icons/Chronometer.png"));
	mBtRecordDuration->setEnabled(false);
	connect(mBtRecordDuration, SIGNAL(clicked()), this, SLOT(recordDuration()) );
	hLayout2->addWidget(mBtRecordDuration);

	mRecordDuration = Duration::seconds(60);

	mBtStop = new QToolButton(this);
	mBtStop->setFixedSize(32,32);
	mBtStop->setAutoRaise(true);
	mBtStop->setToolTip("Stop recording.");
	mBtStop->setIcon(QIcon(":/icons/Stop.png"));
	mBtStop->setEnabled(false);
	connect(mBtStop, SIGNAL(clicked()), this, SLOT(stop()) );
	hLayout2->addWidget(mBtStop);

	hLayout2->addStretch();

	mChkSvTransformsFile = new QCheckBox(this);
	mChkSvTransformsFile->setText("Transform tree file");
	mChkSvTransformsFile->setToolTip("Save all transform links to extra file.");
	mChkSvTransformsFile->setChecked(true);
	connect(mChkSvTransformsFile, SIGNAL(toggled(bool)), this, SLOT(saveTransformsToggled(bool)) );
	hLayout2->addWidget(mChkSvTransformsFile);

	mChkRecTransformsChans = new QCheckBox(this);
	mChkRecTransformsChans->setText("Record transforms");
	mChkRecTransformsChans->setToolTip(
			 QString("Record all transform frame channels. If you prefer to record only\n")
			+QString("specific frames, uncheck this and add your channels individually.") );
	mChkRecTransformsChans->setChecked(true);
	hLayout2->addWidget(mChkRecTransformsChans);

	hLayout2->addStretch();

	mBtAdd = new QPushButton(this);
	mBtAdd->setFixedSize(32,32);
	mBtAdd->setIcon(QIcon(":/icons/Plus.png"));
	mBtAdd->setToolTip("Add a new channel to record.");
	connect(mBtAdd, SIGNAL(clicked()), this, SLOT(add()));
	hLayout2->addWidget(mBtAdd);

	mBtRemove = new QPushButton(this);
	mBtRemove->setFixedSize(32,32);
	mBtRemove->setToolTip("Clear all channels from the list.");
	mBtRemove->setIcon(QIcon(":/icons/ClearList.png"));
	mBtRemove->setEnabled(false);
	connect(mBtRemove, SIGNAL(clicked()), this, SLOT(removeAll()));
	hLayout2->addWidget(mBtRemove);

	mInfo = new QLabel(this);
	hLayout2->addWidget(mInfo);

	vLayout->addLayout(hLayout2);

	connect(&mCheckTimer, SIGNAL(timeout()), this, SLOT(check()));
	connect(&mStatsTimer, SIGNAL(timeout()), this, SLOT(updateStats()));

	mAutoStopTimer.setSingleShot(true);
	connect(&mAutoStopTimer, SIGNAL(timeout()), this, SLOT(stop()));

	setAcceptDrops(true);
}

void TapeRecorderWidget::addChannel(const std::string& channelID, const RecordedChannelInfo& info)
{
	mRecordedChannels.insert(channelID);
	int row = mChannels->rowCount();
	mChannels->insertRow(row);
	mChannels->setItem(row, 1, new QTableWidgetItem(channelID.c_str()));
	mChannels->item(row, 1)->setFlags((Qt::ItemFlags)(Qt::ItemIsSelectable | Qt::ItemIsEnabled));

	CodecWidget* codecWidget = new CodecWidget(this);
	codecWidget->setCodecs(info.codecs);
	mChannels->setCellWidget(row, 2, codecWidget);

	ComboBox* level = new ComboBox(mChannels,
	                               QStringList() << "-1 - Default compression"
	                                             << " 0 - Uncompressed"
	                                             << " 1 - Best speed"
	                                             << " 2" << " 3" << " 4" << " 5" << " 6" << " 7" << " 8"
	                                             << " 9 - Best compression",
	                               QStringList() << "-1" << " 0" << " 1" << " 2" << " 3" << " 4" << " 5" << " 6" << " 7" << " 8" << " 9");
	level->setCurrentIndex(info.compressionLevel+1);
	mChannels->setCellWidget(row, 3, level);
	QSpinBox* edit = new QSpinBox(mChannels);
	edit->setMinimum(0);
	edit->setMaximum(10000000);
	edit->setSingleStep(100);
	edit->setValue(info.interval.totalMilliseconds());
	mChannels->setCellWidget(row, 4, edit);
	mBtRecord->setEnabled(true);
	mBtRecordDuration->setEnabled(true);
	mBtRemove->setEnabled(true);
	QTableWidgetItem* item = new QTableWidgetItem("avoid");
	item->setCheckState(info.avoidDataLoss?Qt::Checked:Qt::Unchecked);
	mChannels->setItem(row, 5, item);
	QPushButton* removeBtn = new QPushButton(mChannels);
	removeBtn->setIcon(QIcon(":/icons/Minus.png"));
	connect(removeBtn, SIGNAL(clicked()), this, SLOT(remove()));
	mChannels->setCellWidget(row, 0, removeBtn);
	mChannels->verticalHeader()->resizeSection(row,24);
}

void TapeRecorderWidget::recordTransforms(bool recordTransformChannels,
                                          bool saveTransformsFile)
{
	mChkSvTransformsFile->setChecked(saveTransformsFile);
	mChkRecTransformsChans->setChecked(recordTransformChannels);
}

void TapeRecorderWidget::add()
{
	SelectionListDialog<std::string> dialog;
	dialog.setHeaderLabels(QStringList("Channels"));
	dialog.setMultiSelection(true);
	std::set<std::string> channelList = MIRA_FW.getChannelManager().getChannels();
	foreach(const std::string& channel, channelList)
		dialog.addItem(QStringList(channel.c_str()), channel);
	if (dialog.exec() == 0)
		return;

	auto selection = dialog.selectedItems();
	if(selection.empty())
		return;

	foreach(auto entry, selection)
		if (mRecordedChannels.count(entry.second) == 0)
			addChannel(entry.second);
}

int TapeRecorderWidget::findButton(int column)
{
	QPushButton* btn = (QPushButton*)sender();
	int row=-1;
	for(int r=0; r<mChannels->rowCount(); ++r)
		if(btn == mChannels->cellWidget(r, column))
		{
			row = r;
			break;
		}
	return row;
}

void TapeRecorderWidget::remove()
{
	int row = findButton(0);
	if (row<0)
		return;
	mRecordedChannels.erase(mChannels->item(row, 1)->text().toLocal8Bit().data());
	mChannels->removeRow(row);
	mBtRecord->setEnabled(mRecordedChannels.size() != 0);
	mBtRecordDuration->setEnabled(mRecordedChannels.size() != 0);
	mBtRemove->setEnabled(mRecordedChannels.size() != 0);
}

void TapeRecorderWidget::removeAll()
{
	for (int i=mChannels->rowCount(); --i >= 0;)
		mChannels->removeRow(i);
	mRecordedChannels.clear();
	mBtRecord->setEnabled(mRecordedChannels.size() != 0);
	mBtRecordDuration->setEnabled(mRecordedChannels.size() != 0);
	mBtRemove->setEnabled(mRecordedChannels.size() != 0);
}

void TapeRecorderWidget::record()
{
	QString fileName = QtUtils::getSaveFileName(this, tr("Filename"), "", "tapes (*.tape)");
	if ( fileName.isEmpty() )
		return;

	mBtRecord->setEnabled(false);
	mBtRecordDuration->setEnabled(false);
	mBtStop->setEnabled(true);
	mChkSvTransformsFile->setEnabled(false);
	mChkRecTransformsChans->setEnabled(false);
	mBtAdd->setEnabled(false);
	mBtRemove->setEnabled(false);
	mChannels->setEnabled(false);

	mRecorder.reset(new TapeRecorder());
	for(int i=0; i<mChannels->rowCount(); ++i)
	{
		std::string channelName = mChannels->item(i, 1)->text().toLocal8Bit().data();
		CodecWidget* codecWidget = dynamic_cast<CodecWidget*>(mChannels->cellWidget(i,2));
		ComboBox* level = dynamic_cast<ComboBox*>(mChannels->cellWidget(i,3));
		QSpinBox* edit = dynamic_cast<QSpinBox*>(mChannels->cellWidget(i,4));
		RecordedChannelInfo info;
		info.codecs = codecWidget->getCodecs();
		info.compressionLevel = level->currentIndex()-1;
		info.avoidDataLoss = mChannels->item(i, 5)->checkState() == Qt::Checked;
		info.interval = Duration::milliseconds(edit->value());
		mRecorder->addChannel(channelName, info);
	}
	if (mChkRecTransformsChans->isChecked())
	{
		auto nodes = MIRA_FW.getTransformer()->getNodes();
		foreach(TransformerBase::AbstractNodePtr node, nodes)
		{
			std::string channelName = node->getID();
			RecordedChannelInfo info;
			info.compressionLevel = 0;
			mRecorder->addChannel(channelName, info);
		}
	}

	emit recordingStarted();
	Path tapeFilename = fileName.toLocal8Bit().data();
	mStartTime = Time::now();
	mRecorder->record(tapeFilename);
	if (mChkSvTransformsFile->isChecked())
		MIRA_FW.getTransformer()->storeTransformTree(tapeFilename.parent_path()/Path(tapeFilename.stem().string()+"_transforms.xml"));
	mCheckTimer.start(100);
	mStatsTimer.start(1000);
}

void TapeRecorderWidget::recordDuration()
{
	DurationDialog dialog(this);
	dialog.setDuration(mRecordDuration);
	if (dialog.exec() == 0)
		return;
	mRecordDuration = dialog.getDuration();

	record();
	if (mCheckTimer.isActive()) {
		uint32 t = std::max<int32>((mStartTime + mRecordDuration - Time::now()).totalMilliseconds(), 0);
		mAutoStopTimer.start(t);
	}
}

void TapeRecorderWidget::stop()
{
	mBtStop->setEnabled(false);
	mCheckTimer.stop();
	mStatsTimer.stop();
	check();
	if (mAutoStopTimer.isActive())
		mAutoStopTimer.stop();

	if (mRecorder)
	{
		mRecorder->stop(false);

		QProgressDialog progress(this);
		progress.setMaximum(mRecorder->storageQueueSize());
		progress.setLabelText("TapeRecorder:\n"
		                      "Processing remaining storage queue\n"
		                      "(writing to tape file)");
		progress.setCancelButton(NULL);

		while (mRecorder->storageQueueSize() > 0) {
			updateStats();
			progress.setValue(progress.maximum() - mRecorder->storageQueueSize());
			QCoreApplication::processEvents();
			mRecorder->processStorageQueue(1);
		}
		// Finally, update the statistics after stopping the recorder to
		// show the number of recorded messages.
		updateStats();
		mRecorder.reset();
	}

	mBtRecord->setEnabled(true);
	mBtRecordDuration->setEnabled(true);
	mChkSvTransformsFile->setEnabled(true);
	mChkRecTransformsChans->setEnabled(true);
	mBtAdd->setEnabled(true);
	mBtRemove->setEnabled(true);
	mChannels->setEnabled(true);
}

void TapeRecorderWidget::check()
{
	if (mRecorder)
	{
		std::stringstream s;
		s << (Time::now()-mStartTime);
		mInfo->setText(s.str().c_str());
	}
}

void TapeRecorderWidget::saveTransformsToggled(bool on)
{
	if (on)
		mChkRecTransformsChans->setChecked(true);
	else
		mChkRecTransformsChans->setChecked(false);
}

void TapeRecorderWidget::dragEnterEvent(QDragEnterEvent *event)
{
	if (mRecorder)
		return;
	const QMimeData* mimeData = event->mimeData();
	if(!mimeData->hasFormat(MIRA_MIME_CHANNEL))
		return;
	QByteArray data = mimeData->data(MIRA_MIME_CHANNEL);
	if ( mRecordedChannels.count(data.data()) > 0 )
		return;
	event->acceptProposedAction();
}

void TapeRecorderWidget::dropEvent(QDropEvent *event)
{
	const QMimeData* mimeData = event->mimeData();
	if(!mimeData->hasFormat(MIRA_MIME_CHANNEL))
		return;
	QByteArray data = mimeData->data(MIRA_MIME_CHANNEL);
	if ( mRecordedChannels.count(data.data()) > 0 )
		return;

	addChannel(data.data());
	event->acceptProposedAction();
}

void TapeRecorderWidget::updateStats()
{
	if(!mRecorder)
		return;

	for(int row=0; row<mChannels->rowCount(); ++row)
	{
		std::string channelName = mChannels->item(row, 1)->text().toLocal8Bit().data();
		const PerformanceStatistics& stats = mRecorder->getStats(channelName);
		mChannels->setItem(row, 6, new QTableWidgetItem(QString::number(stats.counter)));
		mChannels->setItem(row, 7, new QTableWidgetItem(QString("%1").arg(stats.dataPerSecond/1024.0/1024.0, 0, 'f',3)));
	}
}

TapeRecorderWidget::RecordingConfig TapeRecorderWidget::getRecordingConfig()
{
	RecordingConfig config;

	for(int i=0; i<mChannels->rowCount(); ++i)
	{
		std::string channelName = mChannels->item(i, 1)->text().toLocal8Bit().data();
		CodecWidget* codecWidget = dynamic_cast<CodecWidget*>(mChannels->cellWidget(i,2));
		ComboBox* level = dynamic_cast<ComboBox*>(mChannels->cellWidget(i,3));
		QSpinBox* edit = dynamic_cast<QSpinBox*>(mChannels->cellWidget(i,4));
		RecordedChannelInfo info;
		info.codecs = codecWidget->getCodecs();
		info.compressionLevel = level->currentIndex()-1;
		info.interval = Duration::milliseconds(edit->value());
		info.avoidDataLoss = mChannels->item(i, 5)->checkState() == Qt::Checked;
		config.mChannels[channelName] = info;
	}
	config.mRecordTransforms = mChkRecTransformsChans->isChecked();
	config.mSaveTransformsFile = mChkSvTransformsFile->isChecked();
	return config;
}

///////////////////////////////////////////////////////////////////////////////

}
