/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SelectionListDialog.h
 *    Contains the SelectionListDialog widget, a widget for providing the user
 *    a dialog where he/she can choose items from a list, where the items are
 *    grouped in categories.
 *
 * @author Erik Einhorn
 * @date   2010/12/27
 */

#include <QDialog>
#include <QDialogButtonBox>
#include <QVBoxLayout>
#include <QMessageBox>
#include <QTreeWidget>

#include <map>
#include <list>
#include <set>

#include <utils/Foreach.h>

#include <widgets/TreeViewFilter.h>

#ifndef _MIRA_SELECTIONLISTDIALOG_H_
#define _MIRA_SELECTIONLISTDIALOG_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A dialog for providing the a list, where the items can be
 * grouped in categories. The user can choose one or more items from the list.
 *
 * After constructing the dialog and before showing or executing the dialog,
 * items can be added using the addItem() methods. Arbitrary data can be
 * assigned to each item that is added to the list. For this reason, this is
 * a template class. The template parameter must be set to the type of the data
 * that is assigned.
 *
 * After the dialog is shown and the user selected one or more items, the
 * selected items can be retrieved using the selectedItems() method. It returns
 * strings-data pairs, that contain the first label of the selected items and
 * their assigned data.
 *
 * By default multiple selection is disabled. It can be enabled after
 * constructing the dialog by calling the setMultiSelection() method.
 *
 * Example:
 * <code>
 *    SelectionListDialog<int> d(this);
 *    d.setWindowTitle(tr("Choose a something"));
 *    d.setHeaderLabels(QStringList() << "The item" << "Some Description");
 *    d.resize(600,400);
 *
 *    // add item without category and assign 111 as data
 *    d.addItem(QStringList() << "Item 1" << "Description 1", 111);
 *
 *    // add item into a category and assign 222 as data
 *    d.addItem("Some Category", QStringList() << "Item 2" << "Description 2", 222);
 *
 *    // execute as modal dialog
 *    if(d.exec()==0)
 *        return;
 *
 *    auto selection = d.selectedItems();
 *    if(selection.empty())
 *        return; // nothing chosen
 *
 *    // show the assigned int to the data that was chosen (111 or 222)
 *    std::cout << selection.begin()->second << std::endl;
 * </code>
 */
template <typename T>
class SelectionListDialog : public QDialog
{
public:

	SelectionListDialog(QWidget* parent=0);

public:

	/**
	 * Sets the labels of the columns. The number of columns is arbitrary.
	 * However, the number of labels added here must be equal to the number of
	 * labels that is passed to the addItem methods.
	 */
	void setHeaderLabels(const QStringList& labels);

	/**
	 * Adds a new item to the list.
	 * The number of labels must be equal through all addItem calls and
	 * the call to setHeaderLabels. Additional, data can be assigned to each
	 * item. The type of that data must be specified as template parameter of
	 * this template class.
	 */
	void addItem(const QStringList& labels, T data=T(), bool selected=false);

	/**
	 * Adds a new item to the list into the specified category.
	 * The number of labels must be equal through all addItem calls and
	 * the call to setHeaderLabels. Additional, data can be assigned to each
	 * item. The type of that data must be specified as template parameter of
	 * this template class.
	 */
	void addItem(const QString& category, const QStringList& labels, T data=T(),
	             bool selected=false);


public:

	virtual void accept();

	/**
	 * If set to true the user is allowed to only select one item per category.
	 * The selected items are checked when the user clicks on OK and if
	 * more than one item per category is selected an appropriate error message
	 * is shown.
	 */
	void setSelectOnlyOneItemPerCategory(bool onlyOne);

	// Enables or disables multiple item selection, by default it is disabled.
	void setMultiSelection(bool enable);

	/**
	 * Returns a list containing the selected items as pairs of strings and the
	 * data that was assigned in addItem(). As string the first label that was
	 * set in addItem() is given. If the name field is enabled the string is
	 * the string that was entered by the user into the name field.
	 */
	std::list<std::pair<QString, T>> selectedItems() const;

protected:

	QTreeWidgetItem* addCategory(const QString& category);

protected:

	QVBoxLayout* mAdditionalContentLayout;

protected:

	QDialogButtonBox* mButtonBox;
	QTreeWidget*      mTreeWidget;
	bool              mOnlyOnePerCategory;

	std::map<QString, QTreeWidgetItem*> mCategories;
	std::map<const QTreeWidgetItem*, T> mItems;

};

///////////////////////////////////////////////////////////////////////////////

template <typename T>
inline SelectionListDialog<T>::SelectionListDialog(QWidget* parent) :
	QDialog(parent),
	mOnlyOnePerCategory(false)
{
	this->resize(640,480);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	//mainLayout->setSizeConstraint(QLayout::SetFixedSize);

	mTreeWidget = new QTreeWidget(this);
	mTreeWidget->setRootIsDecorated(false);
	mTreeWidget->setAlternatingRowColors(true);


	TreeViewFilter* filter = new TreeViewFilter(mTreeWidget, this);
	filter->setExpandAll(true);
	filter->setFocus();

	mainLayout->addWidget(filter);

	mainLayout->addWidget(mTreeWidget);

	mAdditionalContentLayout = new QVBoxLayout;
	mainLayout->addLayout(mAdditionalContentLayout);

	mButtonBox = new QDialogButtonBox(QDialogButtonBox::Ok |
	                                  QDialogButtonBox::Cancel);

	connect(mButtonBox,  SIGNAL(accepted()), this, SLOT(accept()));
	connect(mButtonBox,  SIGNAL(rejected()), this, SLOT(reject()));
	connect(mTreeWidget, SIGNAL(itemDoubleClicked(QTreeWidgetItem*,int)), this, SLOT(accept()));
	mainLayout->addWidget(mButtonBox);
	setLayout(mainLayout);

}

template <typename T>
inline void SelectionListDialog<T>::setHeaderLabels(const QStringList& labels)
{
	mTreeWidget->setHeaderLabels(labels);
}

template <typename T>
inline QTreeWidgetItem* SelectionListDialog<T>::addCategory(const QString& category)
{
	QTreeWidgetItem* item;

	item = new QTreeWidgetItem(mTreeWidget, QStringList(category));
	mTreeWidget->addTopLevelItem(item);
	QFont font = item->font(0);
	font.setBold(true);
	item->setFont(0,font);
	item->setFlags(Qt::ItemIsEnabled);
	item->setExpanded(true);
	mCategories[category] = item;
	mTreeWidget->resizeColumnToContents(0);
	return item;
}

template <typename T>
inline void SelectionListDialog<T>::addItem(const QString& category,
                                            const QStringList& labels, T data,
                                            bool selected)
{
	assert(!category.isEmpty());

	QTreeWidgetItem* parent = NULL;
	auto it = mCategories.find(category);
	if(it != mCategories.end())
		parent = it->second;
	else {
		parent = addCategory(category);
		mCategories.insert(std::make_pair(category, parent));
	}
	assert(parent!=NULL);

	QTreeWidgetItem* item = new QTreeWidgetItem(parent, labels);
	parent->addChild(item);
	item->setSelected(selected);
	mItems.insert(std::make_pair(item,data));
	mTreeWidget->resizeColumnToContents(0);
}

template <typename T>
inline void SelectionListDialog<T>::addItem(const QStringList& labels, T data,
                                            bool selected)
{
	QTreeWidgetItem* item = new QTreeWidgetItem(labels);
	mTreeWidget->addTopLevelItem(item);
	item->setSelected(selected);
	mItems.insert(std::make_pair(item,data));
	mTreeWidget->resizeColumnToContents(0);
}

template <typename T>
inline void SelectionListDialog<T>::accept()
{
	std::set<QString> itemsPerCategory;
	// check users selection
	if (mOnlyOnePerCategory)
	{
		foreach(const QTreeWidgetItem* i, mTreeWidget->selectedItems()) {
			auto it = mItems.find(i);
			if(it==mItems.end() || i->parent()==NULL)
				continue; // skip
			if (itemsPerCategory.count(i->parent()->text(0)) > 0)
			{
				QMessageBox::critical(this, "Multiple items selected",
				                      "You have selected more than one item per category");
				return;
			}
			itemsPerCategory.insert(i->parent()->text(0));
		}
	}
	QDialog::accept();
}

template <typename T>
inline void SelectionListDialog<T>::setSelectOnlyOneItemPerCategory(bool onlyOne)
{
	mOnlyOnePerCategory = onlyOne;
}

template <typename T>
inline void SelectionListDialog<T>::setMultiSelection(bool enable)
{
	mTreeWidget->setSelectionMode(enable ? QAbstractItemView::MultiSelection :
		QAbstractItemView::SingleSelection);
}

template <typename T>
inline std::list<std::pair<QString, T>> SelectionListDialog<T>::selectedItems() const
{
	std::list<std::pair<QString, T>> r;

	for(QTreeWidgetItemIterator i(mTreeWidget, QTreeWidgetItemIterator::Selected); *i; ++i)
	{
		auto it = mItems.find(*i);
		if(it==mItems.end())
			continue; // skip

		r.push_back(std::make_pair((*i)->text(0), it->second));
	}
	return r;
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
