/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MeshFactory.h
 *    Singleton that allows to load arbitrary meshes.
 *
 * @author Erik Einhorn
 * @date   2011/12/16
 */

#ifndef _MIRA_MESHFACTORY_H_
#define _MIRA_MESHFACTORY_H_

#include <vector>

#ifndef Q_MOC_RUN
#include <OGRE/OgreMesh.h>
#endif

#include <utils/Singleton.h>
#include <visualization/GuiVisualizationExports.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Singleton that allows to load arbitrary meshes from file in different
 * formats.
 * The supported file format depend on the available MeshLoader plugins.
 * Each MeshLoader plugin can handle one or more different mesh file formats.
 * There are three default MeshLoader plugins that can load:
 *  - Ogre-Meshes
 *  - Ogre-Meshes and Materials in zip files
 *  - Stereolithography (STL) files (binary and ascii)
 * More supported formats become available when other MeshLoader plugins
 * are installed (e.g. the Assimp plugin from the Assimp toolbox).
 *
 * In order to use the MeshFactory, manifest files for MeshLoader plugins
 * must have been loaded. E.g. if the program does not instantiate a Framework,
 * it needs to call loadManifests() (ManifestAgent.h) explicitly.
 */
class MIRA_GUI_VISUALIZATION_EXPORT MeshFactory : public LazySingleton<MeshFactory>
{
public:

	/**
	 * Provides information for each format including a human readable
	 * description (e.g. "Stereolithography") and the typical file extension.
	 */
	struct MeshFormatInfo
	{
		MeshFormatInfo() {}

		MeshFormatInfo(const std::string& iDescription,
		               const std::string& iExtension) :
		               description(iDescription), extension(iExtension) {}

		std::string description; ///< the human readable description of the format
		std::string extension;   ///< the file extension
	};

	/**
	 * @return Returns the supported mesh formats that can be loaded with all
	 *         known plugins.
	 */
	std::vector<MeshFormatInfo> getSupportedFormats();

	/**
	 * Loads a mesh from the specified file.
	 * All supported mesh files formats can be queried using getSupportedFormats().
	 *
	 * @throw XIO If the mesh can not be loaded or the format is not supported.
	 */
	Ogre::MeshPtr loadMesh(const std::string& filename);

};

///////////////////////////////////////////////////////////////////////////////

}

#endif
