/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Workbench.h
 *    Declaration of the Workbench.
 *
 * @author Erik Einhorn
 * @date   2010/12/15
 */

#ifndef _MIRA_WORKBENCH_H_
#define _MIRA_WORKBENCH_H_

#include <set>

#include <QSignalMapper>
#include <QMainWindow>

#include <serialization/XMLSerializer.h>

#include <factory/Factory.h>
#include <utils/Buffer.h>

#include <rcp/WorkbenchPart.h>
#include <rcp/ViewPart.h>
#include <rcp/EditorPart.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class PerspectiveTabWidget;

class MIRA_GUI_RCP_EXPORT Workbench : public QMainWindow, public IWorkbenchPartSite
{
	Q_OBJECT
public:

	Workbench(bool setupUi = true);
	virtual ~Workbench();

public:

	void reflect(XMLSerializer& r);
	void reflect(XMLDeserializer& r);

public:

	void setupUi();

public:

	QWidget* createPart(ClassProxy partClass);
	QWidget* restorePart(ClassProxy partClass, const XMLDom& xml);

public:

	virtual void saveWorkspace(const std::string& file);
	virtual void loadWorkspace(const std::string& file);

	/// removes all perspectives
	void cleanupWorkspace();

public:

	class Perspective
	{
	public:
		Perspective() {}
		Perspective(const std::string& name) : mName(name) {}

		const std::string getName() const { return mName; }
		void setName(const std::string& name) { mName = name; }

	public:

		template <typename Reflector>
		void reflect(Reflector& r)
		{
			r.member("Name",  mName,  "");
			r.member("State", mState, "");
		}

	private:
		friend class Workbench;

		std::string mName;
		XMLDom mState; // the state of all views, editors, etc. as XMLDom document
	};
	typedef boost::shared_ptr<Perspective> PerspectivePtr;

private:

	void setupNewViewPart(ViewPart* viewPart);
	void setupNewEditorPart(EditorPart* editorPart);

	void clearWorkspaceParts(); // removes all components of a perspective
	void storeCurrentPerspective();
	void storeCurrentPerspective(Perspective* perspective);
	bool restorePerspective(Perspective* perspective);

public:

	bool isDockAreaMinimized(Qt::DockWidgetArea area);
	void minimizeDockArea(Qt::DockWidgetArea area);
	void restoreDockArea(Qt::DockWidgetArea area);

protected:

	std::list<ViewPart*>   mViewParts;
	std::list<EditorPart*> mEditorParts;

	std::map<std::string, ViewPart*> mCreatedViews;

	std::list<PerspectivePtr> mPerspectives;
	Perspective* mCurrentPerspective;

	struct MinimizedDockArea {
		QDockWidget* dock;
		QWidget* widget;
		QBoxLayout* layout;
		bool minimized;
	};

	std::map<Qt::DockWidgetArea, MinimizedDockArea> mMinimizedDockAreas;
	QSignalMapper* mDockRestoreMapper;

	// state of dockwidget before minimizing, in order to restore its state
	struct MinimizedDockWidgetState {
		QSize size;
		bool visible;
	};
	std::map<QDockWidget*, MinimizedDockWidgetState> mMinimizedDockWidgetStates;

	void createMinimizedDockArea(Qt::DockWidgetArea area);

public: // GUI parts

	QMenu* createPopupMenu();

public slots:

	Perspective* addPerspective(std::string name);
	void removePerspective(Perspective* perspective);
	void activatePerspective(Perspective* perspective);

private slots:

	bool onChangedPerspective(Perspective* perspective);
	void onMovedPerspective(int from, int to);
	void onViewDestroyed(ViewPart* view);
	void onViewActivated(ViewPart* view);
	void onViewMinimizedRequested(ViewPart* view);
	void onViewDockLocationChanged(Qt::DockWidgetArea area);
	void onViewRestore(QWidget* view);

protected slots:
	virtual void onEditorCreated(EditorPart* editor);
	virtual void onEditorDestroyed(EditorPart* editor);

	virtual void onWindowTitleChanged(EditorPartWindow* editor);
	virtual void onEditorWindowActivated(EditorPartWindow* window);

private: // GUI parts

	PerspectiveTabWidget* mPerspectiveTabWidget;

public: // implementation of IWorkbenchPartSite

	virtual void addPartListener(IPartListener* listener);
	virtual void removePartListener(IPartListener* listener);
	virtual IWorkbenchPart* getActiveEditor();
	virtual std::list<IWorkbenchPart*> getEditors(bool backToFront = false);

private:

	std::set<IPartListener*> mPartListeners;
	IWorkbenchPart* mActiveEditorPart;

public:

	void savePartMemory();
	void loadPartMemory();

protected slots:

	void storePartMemory(ViewPart* part);
	void storePartMemory(EditorPart* part);

protected:

	typedef boost::shared_ptr<XMLDom> XMLDomPtr;
	std::map<std::string, XMLDomPtr> mPartMemory;

protected:
	MIRA_DEFINE_EXCEPTION(XInvalidWorkspace, Exception)
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
