/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TestIntPublisher.C
 *
 * @author Tim Langner
 * @date   2010/09/16
 */

#include <fw/Unit.h>

namespace mira { namespace test {

///////////////////////////////////////////////////////////////////////////////

/// Publishes int channel
class TestIntPublisherUnit : public Unit
{
	/// Register at class factory
	MIRA_OBJECT(TestIntPublisherUnit)
public:

	/// Tell base class to call process in a 500 millisecond interval
	TestIntPublisherUnit() :
		Unit(Duration::milliseconds(500))
	{
		mCounter = 0;
	}

	/// Reflect method for serialization and services
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		// call base class reflect
		MIRA_REFLECT_BASE(r, Unit);
		// publish interface ICountSetable
		r.interface("ICountSetable");
		// publish our setCounter message
		r.method("setCounter", &TestIntPublisherUnit::setCounter, this,
		         "sets the internal counter to the specified value",
		         "value", "value to set", 3);

		r.property("TestInt", mIntProperty, "test int property", 2);
		r.roproperty("TestXRuntime",
		             getter(&TestIntPublisherUnit::throwXRuntime, this),
		             "test property throwing XRuntime");
	}

	/// Overwritten from MicroUnit. Gets called once on startup
	virtual void initialize()
	{
		// publish int channel and store it in a member
		mIntChannel = publish<int>("Int");
		// publish our service methods (setCounter)
		publishService(*this);
	}

	/// Overwritten from Unit. Gets called every 500 milliseconds
	virtual void process(const Timer& timer)
	{
		// get write proxy object from int channel
		ChannelWrite<int> value = mIntChannel.write();
		// and write our incremented counter to it
		value->sequenceID = mCounter;
		value->value() = mCounter++;
	}

private:

	int setCounter(int value)
	{
		MIRA_LOG(NOTICE) << "IntPublisherUnit: setCounter called: " << value;
		int oldCount = mCounter;
		mCounter = value;
		return oldCount;
	}

	int throwXRuntime()
	{
		MIRA_THROW(XRuntime, "TestIntPublisherUnit property getter throws XRuntime");
		return 0;
	}

protected:
	Channel<int> mIntChannel;
	int mCounter;
	int mIntProperty;
};

///////////////////////////////////////////////////////////////////////////////

}}

/// Tell class factory that we are derived from Unit
MIRA_CLASS_SERIALIZATION( mira::test::TestIntPublisherUnit, mira::Unit );
