/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MetaSerializationTestUnit.C
 *    Test unit for meta serialization
 *
 * @author Christof Schröter
 * @date   2018/08/20
 */

#define BOOST_TEST_MODULE MetaSerializationPolymorphic
#include <boost/test/unit_test.hpp>
#include <boost/test/execution_monitor.hpp>

#include <fw/Unit.h>
#include <factory/Factory.h>

using namespace mira;

namespace mira { namespace test { namespace meta_serialization {

///////////////////////////////////////////////////////////////////////////////

class DummyBase  : public Object {
	MIRA_OBJECT(DummyBase)
public:
	DummyBase() : f(0), s(toString(0)) {}
	DummyBase(int in) : f(in), s(toString(in)) {}
		
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("f", f, "");
		r.member("s", s, "");
}

public:
	float f;
	std::string s;		
};

typedef boost::shared_ptr<DummyBase> DummyBasePtr;
typedef std::vector<DummyBasePtr> DummyVector;
typedef std::map<int, DummyBasePtr> DummyMap;

class DummyElement {

public:
	DummyElement(int in=0) : i(in) {}
		
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", i, "");
	}

public:
	int i;
};

class DummySub : public DummyBase {
	MIRA_OBJECT(DummySub)
public:
	DummySub() {}
	DummySub(int in) : DummyBase(in) {}
	
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, DummyBase);
		r.member("element", e, "");
	}

public:
	DummyElement e;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Test sender unit for meta serialization with polymorphic types
 */
class PolymorphicTestSender : public Unit
{
	MIRA_OBJECT(PolymorphicTestSender)

public:

	PolymorphicTestSender();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Unit);
		r.property("PublishEmpty", mPublishEmpty,
		           "Whether to publish an empty container first", true);
	}

protected:

	virtual void initialize();

	virtual void process(const Timer& timer);

private:

	Channel<DummyVector> mVecChannel;
	Channel<DummyMap> mMapChannel;
	int mCount;
	bool mPublishEmpty;
};

///////////////////////////////////////////////////////////////////////////////

PolymorphicTestSender::PolymorphicTestSender()
	: Unit(Duration::milliseconds(1000))
{
	mCount = 0;
}

void PolymorphicTestSender::initialize()
{
	mVecChannel = publish<DummyVector>("DummyVector");
	mMapChannel = publish<DummyMap>("DummyMap");
}

void PolymorphicTestSender::process(const Timer& timer)
{
	if (mCount == 0) {
		if (mPublishEmpty) {
			mVecChannel.post(DummyVector()); // post empty container
			mMapChannel.post(DummyMap());
		}
		++mCount;
		return;
	}

	DummyBasePtr p(new DummySub(mCount));

	DummyVector vec;
	vec.push_back(p);
	mVecChannel.post(vec); // create 1 contained element	

	DummyMap map;
	map.insert(std::make_pair(mCount, p));
	mMapChannel.post(map); // create 1 contained element	

	++mCount;
	
}

///////////////////////////////////////////////////////////////////////////////

/**
 * Test receiver unit for meta serialization with polymorphic types
 */
class PolymorphicTestReceiver : public MicroUnit
{
	MIRA_OBJECT(PolymorphicTestReceiver)

public:

	PolymorphicTestReceiver();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, MicroUnit);
	}

protected:

	virtual void initialize();

private:

	void vectorCallback(ChannelRead<void> read);
	void mapCallback(ChannelRead<void> read);

	bool checkMetaDB(const std::string& typeIdentifier);

	Channel<void> mVecChannel;
	Channel<void> mMapChannel;
};

///////////////////////////////////////////////////////////////////////////////

PolymorphicTestReceiver::PolymorphicTestReceiver()
{
}

void PolymorphicTestReceiver::initialize()
{
	mVecChannel = subscribe<void>("DummyVector", &PolymorphicTestReceiver::vectorCallback);
	mMapChannel = subscribe<void>("DummyMap", &PolymorphicTestReceiver::mapCallback);
}

void PolymorphicTestReceiver::vectorCallback(ChannelRead<void> data)
{
	json::Value v;

	try {
		data.readJSON(v);
	}
	catch(XIO& ex) {
		// DispatcherThread::run() will catch any exception anyway, so we just catch to see if BOOST_FAIL
		// actually has any effect (i.e., it throws) - in that case we are in a test case and stop all
		// exceptions here (but still log the errors in the test framework). Otherwise, the user will
		// want to know about this, so we let the original exception pass.
		try {
			BOOST_FAIL("data.readJSON(v) failed");
			throw;
		} catch(boost::execution_aborted) {}
	}
}

void PolymorphicTestReceiver::mapCallback(ChannelRead<void> data)
{
	json::Value v;

	try {
		data.readJSON(v);
	}
	catch(XIO& ex) {
		// DispatcherThread::run() will catch any exception anyway, so we just catch to see if BOOST_FAIL
		// actually has any effect (i.e., it throws) - in that case we are in a test case and stop all
		// exceptions here (but still log the errors in the test framework). Otherwise, the user will
		// want to know about this, so we let the original exception pass.
		try {
			BOOST_FAIL("data.readJSON(v) failed");
			throw;
		} catch(boost::execution_aborted) {}
	}
}

// not used so far
bool PolymorphicTestReceiver::checkMetaDB(const std::string& typeIdentifier)
{
	auto db = MIRA_FW.getMetaDatabase();
	foreach (auto e, *db) {
		if (e.first == typeIdentifier)
			return true;
	}

	return false;
}

///////////////////////////////////////////////////////////////////////////////

}}}

MIRA_CLASS_SERIALIZATION(mira::test::meta_serialization::DummyBase,
                         mira::Object);
MIRA_CLASS_SERIALIZATION(mira::test::meta_serialization::DummySub,
                         mira::test::meta_serialization::DummyBase);
MIRA_CLASS_SERIALIZATION(mira::test::meta_serialization::PolymorphicTestSender,
                         mira::Unit);
MIRA_CLASS_SERIALIZATION(mira::test::meta_serialization::PolymorphicTestReceiver,
                         mira::MicroUnit);
