/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkBugTests.C
 *
 * @author Erik Einhorn
 * @date   2014/01/08
 */

#include <platform/Platform.h>
#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>


using namespace mira;

const char* argv[] = {"FrameworkBasicTest", "-d0", "--no-colors"};
Framework fw(3,(char**)argv);

///////////////////////////////////////////////////////////////////////////////

void ticket622Thread(Channel<int> channel, Time t0)
{
	while(!boost::this_thread::interruption_requested()) {
		std::cout << "Begin Read" << std::endl;
		ChannelReadInterval<int> iv = channel.readInterval(t0, 4, 2,2);
		std::cout << "End Read: " << iv.size() << std::endl;
	}
	std::cout << "End Thread" << std::endl;
}

BOOST_AUTO_TEST_CASE(ticket622Tests)
{
	Authority authority("/", std::string("Authority1"));

	authority.publish<int>("IntervalConcurrentAccess");
	Channel<int> channel = authority.subscribe<int>("IntervalConcurrentAccess", Duration::seconds(100));

	Time t0 = Time(Date(2000, 1, 1), Duration::seconds(0));
	Time t=t0;

	int i;
	// prime the channel with some values
	for(i=0; i<100; ++i)
		channel.post(i, t0+Duration::seconds(i));

	boost::thread otherThread(boost::bind(ticket622Thread, channel, t0));

	for (; i<100000; ++i)
	{
		std::cout << "post " << i << std::endl;
		channel.post(i, t0+Duration::seconds(i));
	}
	std::cout << "End Post" << std::endl;

	otherThread.interrupt();
	if(!otherThread.timed_join(Duration::seconds(1)))
		BOOST_FAIL("Thread ran into an infinite loop within readInterval!");
}

///////////////////////////////////////////////////////////////////////////////


BOOST_AUTO_TEST_CASE(ticket693Tests)
{
	Authority authority("/", std::string("Authority2"));

	Time t0 = Time::now();
	Time t1 = Time::now()+Duration::seconds(1);
	Time t2 = Time::now()+Duration::seconds(2);

	// create untyped channel and generate some data
	{
		Channel<void> untypedChannel = authority.publish<void>("693TestChannel", typeName<float>());
		Buffer<uint8> buffer;
		BinaryBufferSerializer bs(&buffer);
		ChannelWrite<void> w = untypedChannel.write();
		float value = 111.0f;
		w->timestamp = t0;
		bs.serialize(value,false);
		w.writeSerializedValue(buffer);
	}

	// subscribe untyped and obtain a read lock
	Channel<void> untypedChannel = authority.subscribe<void>("693TestChannel");
	ChannelRead<void> readLock = untypedChannel.read();

	// keep the read lock while promoting the channel by publishing typed
	Channel<float> typedChannel = authority.publish<float>("693TestChannel");
	ChannelRead<float> readLock2 = typedChannel.read();

}

///////////////////////////////////////////////////////////////////////////////

bool gGotCallback[10];


void resetCallbackChecks()
{
	std::fill(std::begin(gGotCallback), std::end(gGotCallback), false);
}

void callback(ChannelRead<int> read)
{
	gGotCallback[read->value()] = true;
}

void ticket198Thread(Authority& auth)
{
	MIRA_SLEEP(1000)
//	std::cout << "Thread - unsubscribe" << std::endl;
	auth.unsubscribe<int>("198TestChannel");
//	std::cout << "Thread - unsubscribe done" << std::endl;
	MIRA_SLEEP(1000)
//	std::cout << "End Thread" << std::endl;
}

BOOST_AUTO_TEST_CASE(ticket198Tests)
{
	Authority authority("/", std::string("Authority3"));

	Channel<int> pub = authority.publish<int>("198TestChannel");
	BOOST_CHECK(!pub.hasSubscriber());

	resetCallbackChecks();

	authority.subscribe<int>("198TestChannel", callback);

	BOOST_CHECK(pub.hasSubscriber());

	pub.post(0);
	MIRA_SLEEP(100);
	BOOST_CHECK_EQUAL(gGotCallback[0], true);

	pub.post(1);
	MIRA_SLEEP(100);
	BOOST_CHECK_EQUAL(gGotCallback[1], true);

	boost::thread otherThread(boost::bind(ticket198Thread, boost::ref(authority)));
	otherThread.join();

	// if the channel has a NULL pointer remain in subscribers list (see #198),
	// this test will fail
	BOOST_CHECK(!pub.hasSubscriber());

	pub.post(2);
	MIRA_SLEEP(100);
	BOOST_CHECK_EQUAL(gGotCallback[2], false);
}

///////////////////////////////////////////////////////////////////////////////

