/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <pthread.h>

#include <libplayercore/playercore.h>
#include <libplayerc++/playerc++.h>

#include "../PerfStats.h"

class BenchmarkPublisher : public ThreadedDriver
{
public:

	BenchmarkPublisher(ConfigFile* cf, int section) :
		ThreadedDriver(cf,section,false, PLAYER_MSGQUEUE_DEFAULT_MAXLEN, PLAYER_LASER_CODE)
	{
		total=0;
		count=0;


		size = cf->ReadInt(section, "size", 1);
		std::cout << "Size: " << size << std::endl;

		memset(&mLaserAddr, 0, sizeof(player_devaddr_t));
		cf->ReadDeviceAddr(&(mLaserAddr), section, "provides", PLAYER_LASER_CODE, 0, NULL);

	}

	virtual ~BenchmarkPublisher()
	{
	}

	virtual void Main()
	{
		usleep(2*1000000); // make sure all libs are loaded and mem stats are stable

		uint64_t memStart = getMemoryUsage();
		std::cout << "Publisher Start: " << memStart << std::endl;


		memset(&dataPacket,0,sizeof(dataPacket));
		dataPacket.intensity_count = size*1024;
		dataPacket.intensity = new uint8_t[dataPacket.intensity_count];


		while(true)
		{
			// Check to see if Player is trying to terminate the plugin thread
			pthread_testcancel();

			// Process messages
			ProcessMessages();

			dataPacket.id++;


			uint64_t* t = reinterpret_cast<uint64_t*>(dataPacket.intensity);
			uint64_t start = getCycleCount();
			*t = start;

			Publish(mLaserAddr, PLAYER_MSGTYPE_DATA, PLAYER_LASER_DATA_SCAN, (void*)&dataPacket, 0);

			uint64_t d = getCycleCount() - start;
			std::cout << "Publishing took: " << d << std::endl;

			total+=d;
			count++;

			ProcessMessages();

			uint64_t memDiff2 = getMemoryUsage() - memStart;
			std::cout << "Publisher Memory2: " << memDiff2 << std::endl;

			// Sleep this thread so that it releases system resources to other threads
			usleep(100000);
		}
	}

	virtual void MainQuit()
	{
		std::cout << "Publishing Average: " << total/count << std::endl;
	}

	int size;
	player_laser_data_t dataPacket;
	player_devaddr_t mLaserAddr;

	int count;
	uint64_t total;

	uint64_t memStart;
};

///////////////////////////////////////////////////////////////////////////////



class BenchmarkSubscriber : public ThreadedDriver
{
public:

	BenchmarkSubscriber(ConfigFile* cf, int section) :
		ThreadedDriver(cf,section,false, PLAYER_MSGQUEUE_DEFAULT_MAXLEN, PLAYER_LASER_CODE)
	{
		total=0;
		count=0;

		usleep(5*1000000); // make sure all libs are loaded and mem stats are stable
		memStart = getMemoryUsage();
		std::cout << "Subscriber Start: " << memStart << std::endl;

		if (cf->ReadDeviceAddr(&mLaserAddr, section, "requires", PLAYER_LASER_CODE, -1, NULL) != 0)
		{
			PLAYER_ERROR("Could not find laser interface!");
			SetError(-1);
			return;
		}

		if(!(mLaserDevice = deviceTable->GetDevice(mLaserAddr)))
		{
			PLAYER_ERROR("unable to locate suitable laser device");
			return;
		}

		if(mLaserDevice->Subscribe(this->InQueue) != 0)
		{
			PLAYER_ERROR("unable to subscribe to laser device");
			return;
		}

	}

	virtual ~BenchmarkSubscriber()
	{
	}

	int ProcessMessage(QueuePointer &resp_queue,
	                   player_msghdr * hdr,
	                   void * data)
	{
		player_laser_data_t* dataPacket = reinterpret_cast<player_laser_data_t*>(data);
		uint64_t* t = reinterpret_cast<uint64_t*>(dataPacket->intensity);
		uint64_t d = getCycleCount() - *t;
		std::cout << "Transport took: " << d << std::endl;

		total+=d;
		count++;

		int64_t memDiff2 = getMemoryUsage() - memStart;
		std::cout << "Subscriber Memory2: " << memDiff2 << std::endl;

		return 0;
	}

	virtual void Main()
	{
		while(true)
		{
			// Check to see if Player is trying to terminate the plugin thread
			pthread_testcancel();

			// Process messages
			ProcessMessages();

			// Sleep this thread so that it releases system resources to other threads
			usleep(10);
		}
	}

	virtual void MainQuit()
	{
		std::cout << "Latency Average: " << total/count << std::endl;
	}

	int size;
	player_devaddr_t mLaserAddr;
	Device *mLaserDevice;

	int count;
	uint64_t total;

	int64_t memStart;
};



///////////////////////////////////////////////////////////////////////////////

static Driver* DriverInitPublisher(ConfigFile* cfg, int section)
{
	return (Driver*)(new BenchmarkPublisher(cfg, section));
}

static Driver* DriverInitSubscriber(ConfigFile* cfg, int section)
{
	return (Driver*)(new BenchmarkSubscriber(cfg, section));
}

static void DriverRegister(DriverTable* table)
{
	table->AddDriver("BenchmarkPublisher", DriverInitPublisher);
	table->AddDriver("BenchmarkSubscriber", DriverInitSubscriber);
}

extern "C" {

int player_driver_init(DriverTable* table)
{
	std::cout << "Benchmark_init" << std::endl;
	DriverRegister(table);
	std::cout << "Benchmark_init done" << std::endl;
	return 0;
}

} // extern "C"

