/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */


/**
 * @file ProcessSpawnManager.h
 *
 * @author Erik Einhorn
 * @date   05/09/2012
 */

#ifndef _MIRA_PROCESSSPAWNMANAGER_H_
#define _MIRA_PROCESSSPAWNMANAGER_H_

#include <platform/Process.h>

#include <boost/algorithm/string.hpp>
#include <vector>
#include <list>

#include <error/Logging.h>

#include <utils/Foreach.h>
#include <utils/Path.h>
#include <utils/PathFinder.h>

#include <thread/ThreadMonitor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Implements manager that starts processes and keeps track of them in order
 * to respawn them if required, etc.
 */
class ProcessSpawnManager
{
public:

	struct ProcessInfo
	{
		~ProcessInfo()
		{
			if(!cleanup_cmd.empty() && std::system(NULL)) {
				if (0 != std::system(cleanup_cmd.c_str())) {
					MIRA_LOG(WARNING) << "ProcessSpawnManager: Failed to clean up using command '" << cleanup_cmd << "'";
				}
			}
		}

		std::string getName() const {
			if(name.empty())
				return pid;
			else
				return name;
		}

		Process process;

		std::string name;
		std::string pid;
		bool respawn;
		bool required;

		// infos for respawn
		bool shutdownRecursively;
		Path executable; // the used executable
		std::vector<std::string> args; // arguments of the process
		boost::optional<Process::Environment> env;
		std::string cleanup_cmd;
	};
	typedef boost::shared_ptr<ProcessInfo> ProcessInfoPtr;

	std::list<ProcessInfoPtr> mProcesses;

	boost::thread mWatchdogThread;
	boost::mutex mMutex;

	typedef boost::function<void(const ProcessInfo& p)> RequiredProcessTerminatedFn;

	RequiredProcessTerminatedFn mRequiredProcessTerminatedFn;

public:
	~ProcessSpawnManager()
	{
		mWatchdogThread.interrupt();
		mWatchdogThread.join();
	}

	void startProcess(const Path& executable, const std::vector<std::string>& args,
	                  const std::string& name = "",
	                  bool respawn=false, bool required=false,
	                  bool shutdownRecursively=false,
	                  boost::optional<Process::Environment> env = boost::optional<Process::Environment>(),
	                  const std::string& cleanupCmd="")
	{
		ProcessInfoPtr info(new ProcessInfo);

		// store infos
		info->name = name;
		info->respawn = respawn;
		info->required = required;

		Process::CreationFlags flags = Process::interruptChildOnParentDeath;
		if (shutdownRecursively)
			flags |= Process::shutdownRecursively;

		// launch the new process
		info->process = Process::createProcess(executable.string(), args,
		                                       flags,
		                                       Process::none, env);
		info->pid = toString(info->process.getPID());

		info->shutdownRecursively = shutdownRecursively;
		info->executable = executable;
		info->args = args;
		info->env = env;
		info->cleanup_cmd = cleanupCmd;

		// synchronize
		{
			boost::mutex::scoped_lock lock(mMutex);
			mProcesses.push_back(info);
		}
	}

	void startWatchdog(RequiredProcessTerminatedFn fn = RequiredProcessTerminatedFn())
	{
		mRequiredProcessTerminatedFn = fn;
		// start watchdog thread that respawns processes when they terminate
		// if not yet running
		if(mWatchdogThread.get_id()==boost::thread::id())
			mWatchdogThread = boost::thread(&ProcessSpawnManager::watchdog, this);
	}

	void watchdog()
	{
		ThreadMonitor::instance().addThisThread("#ProcessSpawnManager Watchdog");
		while(!boost::this_thread::interruption_requested()) {
			boost::this_thread::sleep(Duration::seconds(1));

			boost::mutex::scoped_lock lock(mMutex);
			foreach(ProcessInfoPtr info, mProcesses)
			{
				if(info->process.isRunning())
					continue; // process is still running, nothing to do

				MIRA_LOG(NOTICE) << "Process '" << info->getName() << "' "
				                 << "was terminated.";
				// if we reach here, the process was stopped and we need
				// to respawn it or terminate ourselves
				if(info->respawn) {
					Process::CreationFlags flags = Process::interruptChildOnParentDeath;
					if (info->shutdownRecursively)
						flags |= Process::shutdownRecursively;
					MIRA_LOG(NOTICE) << "Respawning: " << info->getName();
					info->process = Process::createProcess(info->executable.string(), info->args,
					                                       flags,
					                                       Process::none, info->env);
					info->pid = toString(info->process.getPID());
				} else if(info->required) {
					if(mRequiredProcessTerminatedFn)
						mRequiredProcessTerminatedFn(*info);
				}
			}
		}
		if(!ThreadMonitor::isDestroyed())
			ThreadMonitor::instance().removeThisThread();
	}

};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
