/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteAuthority.C
 *    Implementation of RemoteAuthority.h.
 *
 * @author Tim Langner, Erik Einhorn, Christof Schröter
 * @date   2011/09/08
 */

#include <fw/RemoteAuthority.h>

#include <algorithm>

#include <serialization/PropertyTree.h>
#include <thread/ThreadMonitor.h>
#include <fw/Framework.h>

namespace mira {

const Duration cRPCTimeout = Duration::seconds(3);
const Duration cPropertiesUpdateInterval = Duration::seconds(1);

///////////////////////////////////////////////////////////////////////////////

/**
 * Implements AbstractRemotePropertyNode.
 * Each RemoteAuthorityPropertyNode instance mirrors a property of the
 * remote Authority. It delegates set/get to the remote Authority via
 * RPC calls to setPropertyJSON and getPropertyJSON.
 */
class RemoteAuthorityPropertyNode : public AbstractRemotePropertyNode
{
public:

	RemoteAuthorityPropertyNode(const PropertyNodeInfo& info) :
		AbstractRemotePropertyNode(info)
	{}

	~RemoteAuthorityPropertyNode()
	{
		// remove all pending updates for us
		RemoteAuthorityRootPropertyNode* root = dynamic_cast<RemoteAuthorityRootPropertyNode*>(getRootNode());

		if (root != NULL)
			root->getUpdater().removeScheduledUpdates(this);
	}

public:
	// implements AbstractRemotePropertyNode
	virtual void setFromJSON(const json::Value& node)
	{
		RemoteAuthorityRootPropertyNode* root = dynamic_cast<RemoteAuthorityRootPropertyNode*>(getRootNode());

		RemoteAuthority* authority = NULL;
		if (root)
			authority = root->getAuthority();

		if (authority == NULL)
			MIRA_THROW(XIO, "Failed to set property, remote authority is not valid.");

		std::string id = fullID(root->getRootProperty());
		RPCFuture<void> res = MIRA_FW.getRPCManager().call<void>(authority->getGlobalID()+"#builtin",
		                                                         "setPropertyJSON", id,
		                                                         node);
		if(!res.timedWait(cRPCTimeout))
			MIRA_THROW(XIO, "Failed to set property '" << id <<
			                "' to remote authority '" << authority->getGlobalID() << "'");
	}

	virtual json::Value getAsJSON() const
	{
		const RemoteAuthorityRootPropertyNode* root = dynamic_cast<const RemoteAuthorityRootPropertyNode*>(getRootNode());

		const RemoteAuthority* authority = NULL;
		if (root)
			authority = root->getAuthority();

		if (authority == NULL)
			MIRA_THROW(XIO, "Failed to get property, remote authority is not valid.");

		// In RemoteAuthority::getProperties() also the root node of the properties
		// will be added. But the root node is not a property. Instead it is the
		// authority itself.
		// We have to avoid doing updates on the root node. Otherwise we're getting
		// errors, since the name of the authority is used as a property name here.
		if (authority->getGlobalID() == id())
			MIRA_THROW(XIO, "Can't get property for the authority itself.")

		// Trigger an update of the property value
		RemoteAuthorityPropertyNode* This = const_cast<RemoteAuthorityPropertyNode*>(this);
		root->getUpdater().scheduleUpdate(authority->getGlobalID(), This);

		// Throw an exception as long as we don't have a property value.
		if (!value)
			MIRA_THROW(XIO, "Waiting for property query")

		if (exception)
			exception->raise();
		 else if (!errorMessage.empty())
			MIRA_THROW(XIO, errorMessage)

		// Return the last known value
		return *value;
	}

	virtual void synchronize()
	{
		MIRA_THROW(XNotImplemented, "RemoteAuthorityPropertyNode::synchronize() not implemented!");
	}

	const PropertyNode* getRootProperty() const
	{
		const RemoteAuthorityRootPropertyNode* root = dynamic_cast<const RemoteAuthorityRootPropertyNode*>(getRootNode());
		if (!root)
			return NULL;

		return root->getRootProperty();
	}

	PropertyNode* getRootProperty()
	{
		const RemoteAuthorityPropertyNode* This = this;
		return const_cast<PropertyNode*>(This->getRootProperty());
	}

	const RemoteAuthority* getAuthority() const
	{
		const RemoteAuthorityRootPropertyNode* root = dynamic_cast<const RemoteAuthorityRootPropertyNode*>(getRootNode());
		if (!root)
			return NULL;

		return root->getAuthority();
	}

	RemoteAuthority* getAuthority()
	{
		const RemoteAuthorityPropertyNode* This = this;
		return const_cast<RemoteAuthority*>(This->getAuthority());
	}


	/// override PropertyNode::beginRemoveChildren
	virtual void beginRemoveChildren(PropertyNodeListenerList& listeners, int index, NodeList::iterator it, int count)
	{
		PropertyNode::beginRemoveChildren(listeners, index, it, count);

		// memorize the nodes being removed
		mChildrenBeingRemoved.clear();
		const NodeList::iterator& start = it;
		NodeList::iterator end = std::next(it, count);
		for (auto i = start; i != end; ++i) {
			RemoteAuthorityPropertyNode* node = dynamic_cast<RemoteAuthorityPropertyNode*>(*i);
			assert(node!=NULL);
			mChildrenBeingRemoved.push_back(node);
		}
	}

	/// override PropertyNode::endRemoveChildren
	virtual void endRemoveChildren(PropertyNodeListenerList& listeners)
	{
		// this should stop adding new updates for the removed nodes
		PropertyNode::endRemoveChildren(listeners);

		// so now remove the ones that were added before
		const RemoteAuthorityRootPropertyNode* remoteRoot = dynamic_cast<const RemoteAuthorityRootPropertyNode*>(getRootNode());
		if (!remoteRoot)
			return;

		foreach (auto node, mChildrenBeingRemoved) {
			remoteRoot->getUpdater().removeScheduledUpdates(node);
		}
	}

protected:
	// need this as PropertyTree's internal structure is private
	struct TreeNodeInfo : public PropertyNodeInfo
	{
		TreeNodeInfo(const PropertyNodeInfo& info) : PropertyNodeInfo(info) {}

		// takes ownership of child!
		void addChild(TreeNodeInfo* child) { children.emplace_back(child); }

		typedef std::vector<boost::shared_ptr<TreeNodeInfo>> NodeList;
		NodeList children;
	};

public:
	void synchronize(const PropertyTree& tree)
	{
		TreeNodeInfo* root = tree.generatePropertyNodes<TreeNodeInfo>();
		synchronize(root);
		delete root;
	}

	void synchronize(const TreeNodeInfo* tree)
	{
		PropertyNode::NodeList::iterator nodeChild = children().begin();
		TreeNodeInfo::NodeList::const_iterator treeChild = tree->children.begin();

		// detect added and removed children
		int index = 0;
		while ((nodeChild != children().end()) || (treeChild != tree->children.end())) {

			if (nodeChild == children().end()) {
				RemoteAuthorityPropertyNode* newNode = new RemoteAuthorityPropertyNode(*(treeChild->get()));
				addChild(newNode);
				nodeChild = children().end();
				++treeChild;
				continue;
			}

			if (treeChild == tree->children.end()) {
				removeChild(index, nodeChild);
				nodeChild = std::next(children().begin(), index);
				continue;
			}

			assert((nodeChild != children().end()) && (treeChild != tree->children.end()));

			if ((*nodeChild)->id() == (*treeChild)->id()) {
				++index;
				++nodeChild;
				++treeChild;
				continue;
			} else {
				// different items at index
				
				// find tree item in other node children
				auto searchNode = std::next(nodeChild);
				for (; searchNode != children().end(); ++searchNode) {
					if ((*treeChild)->id() == (*searchNode)->id())
						break;
				}
				if (searchNode == children().end()) {
					// tree item is not in node -> add item at index
					RemoteAuthorityPropertyNode* newNode = new RemoteAuthorityPropertyNode(*(treeChild->get()));
					addChild(newNode, index);
					++index;
					nodeChild = std::next(children().begin(), index);
					++treeChild;
					continue;
				}

				// find node item in other tree children
				auto searchTree = std::next(treeChild);
				for (; searchTree != tree->children.end(); ++searchTree) {
					if ((*nodeChild)->id() == (*searchTree)->id())
						break;
				}
				if (searchTree == tree->children.end()) {
					// node item is not in tree -> remove item at index
					removeChild(index, nodeChild);
					nodeChild = std::next(children().begin(), index);
					continue;
				}

				// if we end up here, we found 2 different items that are neither removed nor added --> switched order

				int newIndex = std::distance(tree->children.begin(), searchTree) + 1;
				// +1 because it MOVES to back --> 1 element will disappear before the insertion position

				if (newIndex < (int)children().size()) {
					// if there is enough room, move current element to its proper position
					moveChild(index, nodeChild, newIndex, std::next(children().begin(), newIndex));
				} else {
					// else just move it to the back (that would be a valid
					// option even in the first case, just in some cases less efficient)
					moveChild(index, nodeChild, children().size(), children().end());
				}

				nodeChild = std::next(children().begin(), index);
			}
		}

		assert(children().size() == tree->children.size());

		nodeChild = children().begin();
		treeChild = tree->children.begin();

		for (; nodeChild != children().end(); ++nodeChild, ++treeChild) {
				RemoteAuthorityPropertyNode* node = dynamic_cast<RemoteAuthorityPropertyNode*>(*nodeChild);
				assert(node!=NULL);

				node->synchronize(treeChild->get());
		}
	}

public:
	boost::optional<json::Value> value;
	std::unique_ptr<SerializableException> exception;
	std::string errorMessage;

	std::vector<RemoteAuthorityPropertyNode*> mChildrenBeingRemoved;
};

///////////////////////////////////////////////////////////////////////////////

RemoteAuthority::RemoteAuthority(const AuthorityDescription& desc)
	: mRemotePropertiesRoot(this, RemoteAuthorityPropertyUpdater::instance())
{
	mDescription = desc;
	mHasServiceInterfaceQueryMethod = true;
	mHasExtendedPropertyJSONBatchQueryMethod = true;
}

RemoteAuthority::~RemoteAuthority()
{
	if (mRemoteProperties) {
		// TODO: need remove pending updates for mRemoteProperties and all children?
		// or is severing it from the root node enough? (Is p->mParent = NULL; "atomic enough"?)
		mRemotePropertiesRoot.removeAllChildren(); // 'clear' authority, prevent double free
	}
}

StatusManager::StatusMap RemoteAuthority::getStatusMap()
{
	auto future = MIRA_FW.getRPCManager().call<StatusManager::StatusMap>(getGlobalID()+"#builtin",
	                                                                     "getStatusMap");
	if(!future.timedWait(cRPCTimeout))
		return StatusManager::StatusMap();
	return future.get();
}

boost::shared_ptr<PropertyNode> RemoteAuthority::getProperties()
{
	// require a properties update if outdated
	if (mRemoteProperties && !needSynchronizeProperties())
		return mRemoteProperties;

	boost::mutex::scoped_lock lock(mMutex);

	try
	{
		// query properties from remote side
		PropertyTree tree;
		try {
			tree = fetchProperties();
		}
		catch(XIO& ex) {
			return mRemoteProperties; // if query fails, return previous properties, which might be nullptr
		}

		RemoteAuthorityPropertyNode* root = new RemoteAuthorityPropertyNode(tree.getRootNodeInfo());
		tree.generatePropertyNodes<RemoteAuthorityPropertyNode>(root);
		mRemoteProperties.reset(root);
		mRemotePropertiesRoot.addChild(root);

		mRemotePropertiesTimestamp = Time::now();
		mRemotePropertyValuesQueried = false;
		return mRemoteProperties;
	}
	catch(std::exception& ex)
	{
		MIRA_LOG(ERROR) << "Exception while getting remote properties: " << ex.what() << std::endl;
	}
	return mRemoteProperties;
}

bool RemoteAuthority::needSynchronizeProperties()
{
	boost::mutex::scoped_lock lock(mMutex);
	// make sure we don't spend all our time/bandwidth synchronizing the structure and never actually query the values
	return mRemotePropertyValuesQueried && (Time::now() - mRemotePropertiesTimestamp >= cPropertiesUpdateInterval);
}

void RemoteAuthority::synchronizeProperties()
{
	boost::mutex::scoped_lock lock(mMutex);

	try {
		PropertyTree tree = fetchProperties();
		mRemoteProperties->synchronize(tree);
	}
	catch(XIO& ex) {
		return;
	}

	mRemotePropertiesTimestamp = Time::now();
	mRemotePropertyValuesQueried = false;
}

std::set<std::string> RemoteAuthority::getPublishedChannels()
{
	auto future = MIRA_FW.getRPCManager().call<std::set<std::string>>(getGlobalID()+"#builtin",
	                                                                  "getPublishedChannels");
	if(!future.timedWait(cRPCTimeout))
		return std::set<std::string>();
	return future.get();
}

std::set<std::string> RemoteAuthority::getSubscribedChannels()
{
	auto future = MIRA_FW.getRPCManager().call<std::set<std::string>>(getGlobalID()+"#builtin",
	                                                                  "getSubscribedChannels");
	if(!future.timedWait(cRPCTimeout))
		return std::set<std::string>();
	return future.get();
}

std::set<std::string> RemoteAuthority::getServiceInterfaces()
{
	if (!mHasServiceInterfaceQueryMethod)
		return std::set<std::string>();

	auto future = MIRA_FW.getRPCManager().call<std::set<std::string>>(getGlobalID()+"#builtin",
                                                                      "getServiceInterfaces");
	try {
		if(!future.timedWait(cRPCTimeout))
			return std::set<std::string>();
		return future.get();
	}
	catch(XRPC& ex) {
		MIRA_LOG(WARNING) << "Cannot obtain service interfaces from remote authority "
		                  << getGlobalID() << "! Probably the version of the remote framework is too old!";
		mHasServiceInterfaceQueryMethod = false;
		return std::set<std::string>();
	}
}

AbstractAuthority::ChannelNameMapping RemoteAuthority::getPublishedChannelNames()
{
	auto future
		= MIRA_FW.getRPCManager().call<AbstractAuthority::ChannelNameMapping>(getGlobalID()+"#builtin",
		                                                                      "getPublishedChannelNames");
	if(!future.timedWait(cRPCTimeout))
		return AbstractAuthority::ChannelNameMapping();
	return future.get();
}

AbstractAuthority::ChannelNameMapping RemoteAuthority::getSubscribedChannelNames()
{
	auto future
		= MIRA_FW.getRPCManager().call<AbstractAuthority::ChannelNameMapping>(getGlobalID()+"#builtin",
		                                                                      "getSubscribedChannelNames");
	if(!future.timedWait(cRPCTimeout))
		return AbstractAuthority::ChannelNameMapping();
	return future.get();
}

AbstractAuthority::ChannelStatusMap RemoteAuthority::getSubscribedChannelStatus()
{
	if (!MIRA_FW.getRPCManager().implementsInterface(getGlobalID()+"#builtin",
	                                                 "SubscribedChannelStatusProvider")) {
		MIRA_THROW(XRPC, "Remote authority does not provide subscribed channel status");
	}

	auto future
		= MIRA_FW.getRPCManager().call<AbstractAuthority::ChannelStatusMap>(getGlobalID()+"#builtin",
		                                                                    "getSubscribedChannelStatus");
	if(!future.timedWait(cRPCTimeout))
		return AbstractAuthority::ChannelStatusMap();
	return future.get();
}

void RemoteAuthority::start()
{
	auto future = MIRA_FW.getRPCManager().call<void>(getGlobalID()+"#builtin","start");
	if(!future.timedWait(cRPCTimeout))
		MIRA_THROW(XIO, "Failed to start remote authority '" << getGlobalID() << "'");
}

void RemoteAuthority::stop()
{
	auto future = MIRA_FW.getRPCManager().call<void>(getGlobalID()+"#builtin", "stop");
	if(!future.timedWait(cRPCTimeout))
		MIRA_THROW(XIO, "Failed to stop remote authority '" << getGlobalID() << "'");
}

bool RemoteAuthority::hasUnrecoverableFailure() const
{
	auto future = MIRA_FW.getRPCManager().call<bool>(getGlobalID()+"#builtin",
	                                                 "hasUnrecoverableFailure");
	if(!future.timedWait(cRPCTimeout))
		return false; // ToDo: better throw?
	return future.get();
}

bool RemoteAuthority::isStarted() const
{
	auto future = MIRA_FW.getRPCManager().call<bool>(getGlobalID()+"#builtin","isStarted");
	if(!future.timedWait(cRPCTimeout))
		return false; // ToDo: better throw?
	return future.get();
}

bool RemoteAuthority::isRunning() const
{
	auto future = MIRA_FW.getRPCManager().call<bool>(getGlobalID()+"#builtin","isRunning");
	if(!future.timedWait(cRPCTimeout))
		MIRA_THROW(XIO, "Failed to query running state for remote authority '" << getGlobalID() << "'");
	return future.get();
}

PropertyTree RemoteAuthority::fetchProperties()
{
	RPCFuture<PropertyTree> res = MIRA_FW.getRPCManager().call<PropertyTree>(getGlobalID()+"#builtin",
	                                                                         "getProperties");
	if(!res.timedWait(cRPCTimeout))
		MIRA_THROW(XIO, "Timeout on call to " << getGlobalID() << "#builtin.getProperties().");

	return res.get();
}

///////////////////////////////////////////////////////////////////////////////

RemoteAuthorityPropertyUpdater::RemoteAuthorityPropertyUpdater()
{
	mThread = boost::thread(boost::bind(&RemoteAuthorityPropertyUpdater::process, this));
}

RemoteAuthorityPropertyUpdater::~RemoteAuthorityPropertyUpdater()
{
	mThread.interrupt();
	mThread.join();
}

void RemoteAuthorityPropertyUpdater::scheduleUpdate(const std::string& authorityID, RemoteAuthorityPropertyNode* property)
{
	// Add the property to the queue, if it is not included already
	boost::mutex::scoped_lock lock(mMutex);
	for(auto it = mPendingProperties.begin(); it!=mPendingProperties.end(); ++it)
	{
		if(it->second == property)
			return; // property already in queue
	}
	mPendingProperties.push_back(std::make_pair(authorityID, property));
	mSignal.notify_one();
}

void RemoteAuthorityPropertyUpdater::removeScheduledUpdates(RemoteAuthorityPropertyNode* property)
{
	boost::mutex::scoped_lock lock(mMutex);
	auto next = mPendingProperties.begin();
	for(auto it = mPendingProperties.begin(); it!=mPendingProperties.end(); it=next)
	{
		if(it->second == property)
			next = mPendingProperties.erase(it);
		else
			next = it+1;
	}
}

void RemoteAuthorityPropertyUpdater::process()
{
	ThreadMonitor::instance().addThisThread("#RemoteAuthorityPropertyUpdater");

	RemoteAuthority* authority;

	while (!boost::this_thread::interruption_requested())
	{
		try {
			// Get first property from the queue and collect all properties
			// from the same authority
			std::string authorityID;
			std::list<RemoteAuthorityPropertyNode*> properties;
			std::list<std::string> propertiesIDs;

			{
				boost::mutex::scoped_lock lock(mMutex);
				// If there is nothing to do, wait
				if (mPendingProperties.empty()) {
					mSignal.wait(lock);
				}
				if (boost::this_thread::interruption_requested())
					break;

				// still empty? then continue and wait for next signal.
				if (mPendingProperties.empty())
					continue;

				authority = mPendingProperties.front().second->getAuthority();
				if (authority && authority->needSynchronizeProperties()) {
					lock.unlock();
					authority->synchronizeProperties();
					continue;
				}

				authorityID = mPendingProperties.front().first;

				// root node gets severed when RemoteAuthority is destroyed,
				// in that case drop any pending updates for this authorityID
				PropertyNode* root = mPendingProperties.front().second->getRootProperty();

				// collect all pending properties from that authority
				auto next = mPendingProperties.begin();
				for(auto it = mPendingProperties.begin(); it!=mPendingProperties.end(); it=next)
				{
					if(it->first == authorityID) {
						if (root) {
							properties.push_back(it->second);
							propertiesIDs.push_back(it->second->fullID(root));
						}
						next = mPendingProperties.erase(it);
					} else
						next = it+1;
				}
				if (!root)
					continue;
			}

			assert(!propertiesIDs.empty());
			assert(propertiesIDs.size()==properties.size());

			// Get the properties by using an RPC

			// start by trying extended query method (unless we know that failed before)
			if (authority->mHasExtendedPropertyJSONBatchQueryMethod) {
				typedef std::list<PropertyQueryResult> ValueList;
				ValueList res;
				try {
					RPCFuture<ValueList> future
							= MIRA_FW.getRPCManager().call<ValueList>(authorityID+"#builtin",
									                                  "getPropertyJSONBatchExtended", propertiesIDs);
					if(!future.timedWait(cRPCTimeout))
						MIRA_THROW(XIO, "Failed to get properties from remote authority '" << authorityID << "'");

					// check result
					res = future.get(false);
					if(res.size()!=properties.size()) // should never happen, but it's better to be safe
						MIRA_THROW(XLogical, "The size of the returned value list does not match the requested list");
				}
				catch(XRPC& ex) {
					authority->mHasExtendedPropertyJSONBatchQueryMethod = false;
				}

				// Store the return values in the properties
				auto it=properties.begin();
				foreach(PropertyQueryResult& v, res) {
					(*it)->value = v.value;
					(*it)->exception.reset();
					(*it)->errorMessage = v.message;

					if (!v.exception.empty()) {
						try {
							BinaryBufferDeserializer ds(&(v.exception));
							SerializableException* ex;
							ds.deserialize(ex);
							(*it)->exception.reset(ex);
						}
						catch(...) {} // (*it)->exception remains empty, we still have the message
					}
					++it;
				}
			}

			// if extended query method is not available from this authority, use simple query method
			if (!authority->mHasExtendedPropertyJSONBatchQueryMethod) {
				typedef std::list<json::Value> ValueList;
				RPCFuture<ValueList> future
						= MIRA_FW.getRPCManager().call<ValueList>(authorityID+"#builtin",
									                              "getPropertyJSONBatch", propertiesIDs);
				if(!future.timedWait(cRPCTimeout))
					MIRA_THROW(XIO, "Failed to get properties from remote authority '" << authorityID << "'");

				// check result
				try {
					ValueList res = future.get();
					if(res.size()!=properties.size()) // should never happen, but it's better to be safe
						MIRA_THROW(XLogical, "The size of the returned value list does not match the requested list");

					// Store the return values in the properties
					auto it=properties.begin();
					foreach(json::Value& v, res) {
						(*it)->value = v;
						(*it)->exception.reset();
						(*it)->errorMessage.clear();

						// with the old implementation before getPropertyJSONBatchExtended exists,
						// when a property getter throws an exception other than XInvalidParameter,
						// we will not even get the null JSON value, but the RPC call will just throw the exception
						// (see catch below). so this is the 'lucky' special case
						if (v.is_null())
							(*it)->errorMessage = "Query returned null value (check remote log for details)";

						++it;
					}
				}
				catch(...) { // old implementation of AuthorityManager::getPropertyJSONBatch only
				             // catches XInvalidParameter -> any other exception thrown on any property getter
				             // in the queried batch will fail the query for the entire batch
					foreach(auto& p, properties) {
						p->value = json::Value();
						p->exception.reset();
						p->errorMessage = "Query of property batch failed, value could not be updated";
					}
				}
			}

			authority->mRemotePropertyValuesQueried = true;

		} catch(std::exception& ex) {
			MIRA_LOG(ERROR) << "Exception while obtaining remote properties: " << ex.what() << std::endl;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
