/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AuthorityManager.C
 *    Implementation of AuthorityManager.h.
 *
 * @author Erik Einhorn
 * @date   2010/10/07
 */

#include <fw/Framework.h>
#include <fw/AuthorityManager.h>

#include <error/Logging.h>
#include <utils/Foreach.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

struct AuthorityService
{
	AuthorityService() : authority(NULL) {}
	AuthorityService(Authority* a) : authority(a) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.method("start", &Authority::start, authority,
		         "Starts the authority, if it was stopped before");
		r.method("stop", &Authority::stop, authority,
		         "Stops the authority.");
		r.method("isRunning", &Authority::isRunning, authority,
		         "Is the authority's main dispatcher running.");
		r.method("isStarted", &Authority::isStarted, authority,
		         "Is the authority started via start()");
		r.method("hasUnrecoverableFailure", &Authority::hasUnrecoverableFailure, authority,
		         "Does the authority has an error state that is unrecoverable.");

		r.method("resolveName", &Authority::resolveName, authority,
		         "Converts the given name into a fully qualified name and returns it.",
		         "name", "input name", "MyAuthority");
		r.method("getPublishedChannels", &Authority::getPublishedChannels, authority,
		         "All channels published by this authority and all its sub-authorities");
		r.method("getSubscribedChannels", &Authority::getSubscribedChannels, authority,
		         "All channels subscribed to by this authority and all its sub-authorities");
		r.method("getServiceInterfaces", &Authority::getServiceInterfaces, authority,
				 "All service interfaces provided by this authority and all its sub-authorities");
		r.method("getPublishedChannelNames", &Authority::getPublishedChannelNames, authority,
		         "All mappings of global to local names for channels published by this authority"
		         " and all its sub-authorities");
		r.method("getSubscribedChannelNames", &Authority::getSubscribedChannelNames, authority,
		         "All mappings of global to local names for channels subscribed to by this "
		         "authority and all its sub-authorities");

		r.interface("SubscribedChannelStatusProvider");
		r.method("getSubscribedChannelStatus", &Authority::getSubscribedChannelStatus, authority,
		         "Current status for all channels subscribed to by this authority and all its sub-authorities");

		r.method("getStatusMap", &Authority::getStatusMap, authority,
		         "Return the status of the authority and all its sub-authorities");
		r.method("getDescription", &AbstractAuthority::getDescription, static_cast<mira::AbstractAuthority*>(authority),
		         "Returns the description of the authority");

		r.method("getProperty", &Authority::getProperty, authority,
		         "Returns the value of the specified property as string",
		         "property", "property specifier", "MainControlUnit.UseOdometryCorrection");
		r.method("setProperty", &Authority::setProperty, authority,
		         "Sets the value of the specified property from string",
		         "property", "property name", "MainControlUnit.UseOdometryCorrection",
		         "value", "property value", "true");
		r.method("getPropertyJSON", &Authority::getPropertyJSON, authority,
		         "Returns the value of the specified property as JSON value",
		         "property", "property name", "MainControlUnit.UseOdometryCorrection");
		r.template method<std::list<json::Value>, const std::list<std::string>&>("getPropertyJSONBatch",
		         boost::bind(AuthorityService::getPropertyJSONBatch, authority, _1),
		         "Returns a list of values of the specified list of properties as JSON values",
		         "properties", "property names",
		         std::list<std::string>({"MainControlUnit.UseOdometryCorrection", "Charger.AutoPowerOn"}));
		r.template method<std::list<PropertyQueryResult>, const std::list<std::string>&>("getPropertyJSONBatchExtended",
		         boost::bind(AuthorityService::getPropertyJSONBatchExtended, authority, _1),
		         "Returns a list of values of the specified list of properties as JSON values, "
		         "includes info on exceptions thrown during query",
		         "properties", "property names",
		         std::list<std::string>({"MainControlUnit.UseOdometryCorrection", "Charger.AutoPowerOn"}));
		r.method("setPropertyJSON", &Authority::setPropertyJSON, authority,
		         "Sets the value of the specified property from JSON value",
		         "property", "property name", "MainControlUnit.UseOdometryCorrection",
		         "value", "property value", json::Value(true));
		r.template method<void, const std::map<std::string, json::Value>&>("setPropertyJSONBatch",
		         boost::bind(AuthorityService::setPropertyJSONBatch, authority, _1),
		         "Sets a list of properties to the specified JSON values",
		         "properties", "property names and values",
		         std::map<std::string, json::Value>({{"MainControlUnit.UseOdometryCorrection", json::Value(true)},
		                                             {"Charger.AutoPowerOn", json::Value(false)}}));
		r.method("getProperties", &Authority::getPropertiesRPC, authority,
		         "Returns the property tree of the authority");
	}

	Authority* authority;

	static std::list<json::Value> getPropertyJSONBatch(Authority* authority, const std::list<std::string>& properties)
	{
		if(!authority)
			MIRA_THROW(XLogical, "No authority set");

		std::list<json::Value> res;
		// if there is an exception on one property, only miss the value for that one property!
		foreach(const std::string& p, properties) {
			try {
				res.emplace_back(authority->getPropertyJSON(p));
			}
			catch (std::exception& ex) {
				MIRA_LOG_EXCEPTION(ERROR, ex) << "Exception in getPropertyJSONBatch, "
				                              << "while getting property '"
				                              << authority->getID() << "." << p << "': ";
				res.emplace_back();
			}
			catch (...) {
				MIRA_LOG(ERROR) << "Unknown exception in getPropertyJSONBatch, "
				                << "while getting property '"
				                << authority->getID() << "." << p << "'";
				res.emplace_back();
			}
		}
		return res;
	}

	static std::list<PropertyQueryResult>
	getPropertyJSONBatchExtended(Authority* authority, const std::list<std::string>& properties)
	{
		if(!authority)
			MIRA_THROW(XLogical, "No authority set");

		std::list<PropertyQueryResult> res;
		// if there is an exception on one property, only miss the value for that one property!
		foreach(const std::string& p, properties) {
			try {
				res.push_back(PropertyQueryResult{authority->getPropertyJSON(p), Buffer<uint8>(), std::string()});
			}
			catch (SerializableException& ex) {
				MIRA_LOG_EXCEPTION(ERROR, ex) << "Exception in getPropertyJSONBatch, "
				                              << "while getting property '"
				                              << authority->getID() << "." << p << "': ";

				Buffer<uint8> buf;
				BinaryBufferSerializer bs(&buf);
				bs.serialize(&ex);
				res.push_back(PropertyQueryResult{json::Value(), std::move(buf), ex.message()});
			}
			catch (Exception& ex) {
				MIRA_LOG_EXCEPTION(ERROR, ex) << "Exception in getPropertyJSONBatch, "
				                              << "while getting property '"
				                              << authority->getID() << "." << p << "': ";
				res.push_back(PropertyQueryResult{json::Value(), Buffer<uint8>(), ex.message()});
			}
			catch (std::exception& ex) {
				MIRA_LOG_EXCEPTION(ERROR, ex) << "Exception in getPropertyJSONBatch, "
				                              << "while getting property '"
				                              << authority->getID() << "." << p << "': ";
				res.push_back(PropertyQueryResult{json::Value(), Buffer<uint8>(), ex.what()});
			}
			catch (...) {
				MIRA_LOG(ERROR) << "Unknown exception in getPropertyJSONBatch, "
				                << "while getting property '"
				                << authority->getID() << "." << p << "'";
				res.push_back(PropertyQueryResult{json::Value(), Buffer<uint8>(), std::string("Unknown exception")});
			}
		}
		return res;
	}

	static void setPropertyJSONBatch(Authority* authority, const std::map<std::string, json::Value>& properties)
	{
		if(!authority)
			MIRA_THROW(XLogical, "No authority set");

		std::list<json::Value> res;
		foreach(const auto p, properties)
		{
			const std::string& property = p.first;
			const json::Value& value = p.second;

			authority->setPropertyJSON(property, value);
		}
	}
};

///////////////////////////////////////////////////////////////////////////////

AuthorityManager::AuthorityManager() :
	mAuthority("/", "AuthorityManager", Authority::ANONYMOUS | Authority::INTERNAL)
{
}

AuthorityManager::~AuthorityManager()
{
	// we need to checkout our own authority first
	mAuthority.checkout();

	MIRA_LOG(DEBUG) << "Authorities remaining: " << mAuthorities.size();
	if(!mAuthorities.empty()) {
		std::string s;
		foreach(const AuthorityMap::value_type& i, mAuthorities)
			s += i.first + ", ";

		MIRA_LOG(ERROR) << "The following authorities are still existing "
		                   "while destroying the AuthorityManager: " << s;
	}
}

void AuthorityManager::addAuthority(AbstractAuthority* authority)
{
	assert(authority != NULL);
	{	// scope for mutex protecting mAuthorities
		boost::mutex::scoped_lock lock(mAuthoritiesMutex);
		// make sure that authority does not exist yet,
		// if so, destroy it and raise exception
		if(mAuthorities.count(authority->getGlobalID())!=0) {
			MIRA_THROW(XInvalidParameter, "The authority '"
			           << authority->getGlobalID() << "' already exists");
		}

		MIRA_LOG(DEBUG) << "Adding authority '" << authority->getGlobalID() << "'";
		mAuthorities[authority->getGlobalID()].reset(new LockableAuthority(authority));
	}

	Authority* localAuthority = dynamic_cast<Authority*>(authority);
	if(localAuthority!=NULL && 
		localAuthority!=&mAuthority &&
		!localAuthority->mIsInternal) {
		AuthorityService s(localAuthority);
		localAuthority->publishService(localAuthority->getGlobalID()+"#builtin", s,
		                               mAuthority.mRPCHandler);
	}
}

// making our own scoped 2-lock class here is simpler than try-catch around boost::lock(m1, m2)
template <typename MutexType1, typename MutexType2>
class ScopedLock2
{
public:
	ScopedLock2(MutexType1& m1, MutexType2& m2) : m1(m1), m2(m2) { boost::lock(m1, m2); }
	~ScopedLock2() { m1.unlock(); m2.unlock(); }
private:
	MutexType1& m1;
	MutexType2& m2;
};

template <typename MutexType1, typename MutexType2>
ScopedLock2<MutexType1, MutexType2> makeScopedLock2(MutexType1& m1, MutexType2& m2)
{
	return ScopedLock2<MutexType1, MutexType2>(m1, m2);
}

void AuthorityManager::removeAuthority(AbstractAuthority* authority)
{
	assert(authority != NULL);
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(authority->getGlobalID());
	if(it == mAuthorities.end()) {
		MIRA_LOG(WARNING) << "The authority '" << authority->getGlobalID()
		                  << "' does not exist";
		return;
	}

	auto authorityPtr = it->second;
	lock.unlock();

	// ensure that both the mAuthoritiesMutex and the authorityPtr->mutex are acquired TOGETHER
	// - making sure two threads can not block each other by each holding one of the mutexes

	// if for some reason removeAuthority is running twice in parallel, actually removing the
	// authority from mAuthorities is synched (only done once).
	// access to authorityPtr->mutex is safe: authorityPtr is a shared ptr, so the mutex
	// exists as long as we hold the pointer, even after executing mAuthorities.erase(it).

	auto lock2 = makeScopedLock2(authorityPtr->mutex, mAuthoritiesMutex);

	it = mAuthorities.find(authority->getGlobalID());
	if(it != mAuthorities.end()) {
		MIRA_LOG(DEBUG) << "Removing authority '" << authority->getGlobalID() << "'";
		mAuthorities.erase(it);
	}
}

bool AuthorityManager::hasAuthority(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	return mAuthorities.count(id)!=0;
}

void AuthorityManager::start(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	it->second->authority->start();
}

void AuthorityManager::stop(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	it->second->authority->stop();
}

bool AuthorityManager::isRunning(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		return false;
	return it->second->authority->isRunning();
}

bool AuthorityManager::isLocal(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	Authority* local = dynamic_cast<Authority*>(it->second->authority);
	return local != NULL;
}

bool AuthorityManager::hasUnrecoverableFailure(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	return it->second->authority->hasUnrecoverableFailure();
}

StatusManager::StatusMap AuthorityManager::getStatusMap(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	return it->second->authority->getStatusMap();
}

AuthorityDescription AuthorityManager::getDescription(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	return it->second->authority->getDescription();
}

std::set<std::string> AuthorityManager::getPublishedChannels(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	return it->second->authority->getPublishedChannels();
}

std::set<std::string> AuthorityManager::getSubscribedChannels(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	return it->second->authority->getSubscribedChannels();
}

boost::shared_ptr<PropertyNode> AuthorityManager::getProperties(const std::string& id) const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	return it->second->authority->getProperties();
}

std::set<std::string> AuthorityManager::getAuthorities() const
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	std::set<std::string> ret;
	foreach(const AuthorityMap::value_type& i, mAuthorities)
	{
		Authority* local = dynamic_cast<Authority*>(i.second->authority);
		if (!local || !local->mIsInternal)
			ret.insert(i.second->authority->getGlobalID());
	}
	return ret;
}

SharedAuthority AuthorityManager::getAuthority(const std::string& id)
{
	boost::mutex::scoped_lock lock(mAuthoritiesMutex);
	auto it = mAuthorities.find(id);
	if (it == mAuthorities.end())
		MIRA_THROW(XInvalidParameter, "The authority '"
		           << id << "' does not exist");
	return SharedAuthority(it->second->authority, it->second->mutex);
}

///////////////////////////////////////////////////////////////////////////////

}
