/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file NameRegistry.h
 *    Resolving names of channels and authorities in namespaces.
 *
 * @author Erik Einhorn
 * @date   2010/09/13
 */

#ifndef _MIRA_NAMEREGISTRY_H_
#define _MIRA_NAMEREGISTRY_H_

#include <map>

#ifndef Q_MOC_RUN
#include <boost/algorithm/string.hpp>
#endif

#include <fw/ResourceName.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Stores aliases for namespaces and allows to resolve
 * local names to global fully qualified names.
 */
class MIRA_FRAMEWORK_EXPORT NameRegistry
{
public:

	typedef std::map<ResourceName, std::pair<ResourceName, std::string>> AliasMap;

public:

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("INameRegistry");
		r.method("getAliases", &NameRegistry::getAliases, this,
		         "Query aliases ('using' declarations)");
	}

	/**
	 * Sets an alias that is valid for the whole namespace
	 * that is specified.
	 */
	void addAlias(const ResourceName& from, const ResourceName& to,
	              const ResourceName& ns, const std::string& annotation = "");

	/**
	 * Resolves the specified name within the specified namespace and returns
	 * the fully qualified global name.
	 */
	ResourceName resolve(const ResourceName& name, const ResourceName& ns) const;

	/**
	 * Query list of aliases.
	 */
	const AliasMap& getAliases() const;

private:

	AliasMap mAliases;

};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
