/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkGraph.h
 *    Graph representation of channels and authorities of frameworks
 *
 * @author Erik Einhorn
 * @date   2011/03/08
 */

#ifndef _MIRA_FRAMEWORKGRAPH_H_
#define _MIRA_FRAMEWORKGRAPH_H_

#include <string>

#include <serialization/adapters/std/list>
#include <serialization/adapters/boost/shared_ptr.hpp>
#include <geometry/Point.h>

#include <fw/AuthorityDescription.h>

#include <fw/Framework.h>
#include <fw/RemoteModule.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <int C, typename T>
class FrameworkGraphMixin : public T {};

template <int C> // note, the C helps us to generate different classes to avoid compiler warning concerning inaccessible base classes
class FrameworkGraphMixin<C,void> {};

/**
 * Represents the whole computation graph with all known units/authorities
 * and channels within their namespaces.
 * Each such component is represented as graph node. For each such node
 * a separate class is used. For each node class, the user can specify
 * "mixin" classes to add additional members to the nodes. The Mixins are
 * specified as template parameters:
 * - NodeMixin: Mixin for all nodes
 * - AuthorityNodeMixin: Mixin used for authority nodes
 * - ChannelNodeMixin: Mixin used for channel nodes
 * - NamespaceNodeMixin: Mixin used for namespace nodes
 * - FrameworkNodeMixin: Mixin used for framework nodes
 *
 * After you created an instance of this FrameworkGraph class, you can call
 * the discover() method to collect all information about the framework
 * computation graph. The discover() method can also be called multiple
 * times to update the graph and the contained nodes.
 *
 * Note that this class is used to collect the information on the computation
 * network graph only. It is used for visualization purposes and to obtain
 * the overall structure of the computation graph in a "human comprehensible
 * format". This class is NOT intended to store any information that is required
 * by the runtime components! Internally, all the information is distributed
 * in different components.
 */
template <typename NodeMixin = void,
          typename AuthorityNodeMixin=void,
          typename ChannelNodeMixin=void,
          typename NamespaceNodeMixin=void,
          typename FrameworkNodeMixin=void>
class FrameworkGraph : boost::noncopyable
{
	class EmptyMixin {};

public:

	class Node;
	typedef boost::shared_ptr<Node> NodePtr;

	class ChannelNode;
	typedef boost::shared_ptr<ChannelNode> ChannelNodePtr;

	class AuthorityNode;
	typedef boost::shared_ptr<AuthorityNode> AuthorityNodePtr;

	class NamespaceNode;
	typedef boost::shared_ptr<NamespaceNode> NamespaceNodePtr;

	class FrameworkNode;
	typedef boost::shared_ptr<FrameworkNode> FrameworkNodePtr;

	/**
	 * Base class for all nodes in the framework computation graph.
	 */
	class Node : public FrameworkGraphMixin<0,NodeMixin>
	{
	public:
		virtual ~Node() {}

		/// Returns the adjacent (incoming and outgoing nodes)
		virtual std::set<NodePtr> getAdjacentNodes() = 0;
	};

	/**
	 * Represents a single channel in the framwork computation graph.
	 */
	class ChannelNode : public Node, public FrameworkGraphMixin<1,ChannelNodeMixin>
	{
	public:
		ChannelNode(const std::string& iID) : id(iID) {}
		bool operator==(const std::string& id) {
			return id==this->id;
		}

		virtual std::set<NodePtr> getAdjacentNodes()
		{
			std::set<NodePtr> nodes;
			foreach(auto p, mSubscribedAuthorities)
				nodes.insert(p);
			foreach(auto p, mPublishingAuthorities)
				nodes.insert(p);
			return nodes;
		}

		std::set<AuthorityNodePtr> getSubscribedAuthorities() {
			return mSubscribedAuthorities;
		}

		std::set<AuthorityNodePtr> getPublishingAuthorities() {
			return mPublishingAuthorities;
		}

		/// Returns the namespace node where this channel lives in
		NamespaceNodePtr getNamespace() {
			return mNamespace;
		}

	public:

		/// the fully qualified name of the node (the id)
		std::string id;

	protected:
		friend class FrameworkGraph;
		std::set<AuthorityNodePtr> mSubscribedAuthorities;
		std::set<AuthorityNodePtr> mPublishingAuthorities;
		NamespaceNodePtr mNamespace;
	};

	/**
	 * Represents a single authority in the framwork computation graph.
	 */
	class AuthorityNode : public Node, public FrameworkGraphMixin<2,AuthorityNodeMixin> {
	public:
		AuthorityNode(const AuthorityDescription& iDesc) : desc(iDesc) {}
		bool operator==(const std::string& id) {
			return id == (desc.ns / desc.id);
		}

	public:

		virtual std::set<NodePtr> getAdjacentNodes()
		{
			std::set<NodePtr> nodes;
			foreach(auto p, mSubscribedChannels)
				nodes.insert(p);
			foreach(auto p, mPublishedChannels)
				nodes.insert(p);
			return nodes;
		}

		/// Returns all channel nodes with channels that are published by the authority.
		std::set<ChannelNodePtr> getPublishedChannels() {
			return mPublishedChannels;
		}

		/// Returns all channel nodes with channels the authority is subscribed on.
		std::set<ChannelNodePtr> getSubscribedChannels() {
			return mSubscribedChannels;
		}

		/// Returns the namespace node where this authority lives in
		NamespaceNodePtr getNamespace() {
			return mNamespace;
		}

		/// Returns the framework where this authority lives in
		FrameworkNodePtr getFramework() {
			return mFramework;
		}

		/**
		 * Returns all authority nodes that THIS authority depends on.
		 * For each such authority a set of channels is given, that cause the
		 * dependency.
		 */
		std::map<AuthorityNodePtr, std::set<ChannelNodePtr>> getDependees() {
			std::map<AuthorityNodePtr, std::set<ChannelNodePtr>> nodes;
			foreach(auto c, mSubscribedChannels)
				foreach(auto a, c->getPublishingAuthorities())
					nodes[a].insert(c);
			return nodes;
		}

		/**
		 * Returns all authority nodes that DEPEND on this authority.
		 * For each such authority a set of channels is given, that cause the
		 * dependency.
		 */
		std::map<AuthorityNodePtr, std::set<ChannelNodePtr>> getDependers() {
			std::map<AuthorityNodePtr, std::set<ChannelNodePtr>> nodes;
			foreach(auto c, mPublishedChannels)
				foreach(auto a, c->getSubscribedAuthorities())
					nodes[a].insert(c);
			return nodes;
		}


	public:

		/// full information on the authority
		AuthorityDescription desc;

	protected:
		friend class FrameworkGraph;
		std::set<ChannelNodePtr> mSubscribedChannels;
		std::set<ChannelNodePtr> mPublishedChannels;
		NamespaceNodePtr mNamespace;
		FrameworkNodePtr mFramework;
	};

	/**
	 * Represents a namespace in the framework computation graph within
	 * the overall namespace hierarchy.
	 */
	class NamespaceNode : public Node, public FrameworkGraphMixin<3,NamespaceNodeMixin> {
	public:
		NamespaceNode(const ResourceName& iNs) : ns(iNs) {}
		bool operator==(const ResourceName& iNs) {
			return iNs==ns;
		}

		/// workaround to avoid shared_from_this
		static void setParent(NamespaceNodePtr child, NamespaceNodePtr parent) {
			assert(parent && child);
			parent->children.insert(child);
			child->parent = parent;
		}

	public:

		virtual std::set<NodePtr> getAdjacentNodes()
		{
			static std::set<NodePtr> empty;
			return empty;
		}

	public:

		/// the name of the namespace
		ResourceName ns;

		/// the namespace parent node (if we are not the root node)
		NamespaceNodePtr parent;

		/// sub-namespaces that live in this namespace
		std::set<NamespaceNodePtr> children;

		/// channel and authority nodes within this namespace
		std::set<NodePtr> nodes;
	};

	/**
	 * Represents a framework in the framework computation graph
	 */
	class FrameworkNode : public Node, public FrameworkGraphMixin<4,FrameworkNodeMixin> {
	public:
		FrameworkNode(const std::string& iName) : name(iName) {}
		bool operator==(const std::string& iName) {
			return iName==name;
		}

	public:

		virtual std::set<NodePtr> getAdjacentNodes()
		{
			static std::set<NodePtr> empty;
			return empty;
		}

	public:

		/// the name
		std::string name;

		/// channel and authority nodes within this framework
		std::set<NodePtr> nodes;
	};

public:

	/// Creates an emtpy framework graph object.
	FrameworkGraph() {
		// create a root namespace node
		mRootNamespace = getNamespaceNode(ResourceName("/"));
	}


	FrameworkGraph(FrameworkGraph&& other) noexcept {
		swap(other);
	}

	FrameworkGraph& operator=(FrameworkGraph&& other) noexcept {
		swap(other);
		return *this;
	}

	void swap(FrameworkGraph& other)
	{
		boost::swap(mMutex, other.mMutex);
		std::swap(mAuthorities, other.mAuthorities);
		std::swap(mChannels, other.mChannels);
		std::swap(mNamespaces, other.mNamespaces);
		std::swap(mFrameworks, other.mFrameworks);
		std::swap(mRootNamespace, other.mRootNamespace);
		std::swap(mGraphHasChanged, other.mGraphHasChanged);
	}

	boost::mutex& mutex() { return mMutex; }

	/**
	 * Discovers the whole structure of all connected frameworks with its
	 * authorities, channels, services, etc.
	 * Returns true, if the graph has changed, otherwise false.
	 * The optional parameter allows to specify a callback that is called
	 * from time to time during the discovery process, whenever changes in the
	 * graph have been observed.
	 */
	bool discover(boost::function<void()> changeCb = boost::function<void()>())
	{
		mGraphHasChanged = false; // <this variable is reseted several times
		bool graphHasChanged = false; // <this one holds whether the graph has any changes

		// Channels
		foreach(const std::string& id, MIRA_FW.getChannelManager().getChannels())
		{
			ChannelNodePtr node = getChannelNode(id);
		}

		graphHasChanged |= mGraphHasChanged;
		if(mGraphHasChanged && changeCb)
			changeCb();

		// Authorities
		// discovery of authorities may be a little time consuming, if
		// a lot of remote authorities and hence RPC calls are involved
		auto authorities = MIRA_FW.getAuthorityManager().getAuthorities();
		foreach(auto id, authorities)
		{
			mGraphHasChanged = false; // reset marker

			AuthorityNodePtr node = findPtr(mAuthorities, id);
			if(!node) {
				AuthorityDescription d = MIRA_FW.getAuthorityManager().getDescription(id);

				boost::mutex::scoped_lock lock(mutex());
				node.reset(new AuthorityNode(d));
				mAuthorities.push_back(node);

				// Namespace
				NamespaceNodePtr ns = getNamespaceNode(d.ns);
				node->mNamespace = ns;
				ns->nodes.insert(node);

				// Framework
				std::string fwid = "local";
				foreach(const auto& connection, *(MIRA_FW.getRemoteModule()->getConnections()))
				{
					if (connection.second->hasAuthority(id)) {
						fwid = connection.second->getAddress().address;
						break;
					}
				}
				FrameworkNodePtr fw = getFrameworkNode(fwid);
				node->mFramework = fw;
				fw->nodes.insert(node);

				mGraphHasChanged = true;
			}

			// connections: Authorities -> Channels:
			auto publishedChannels = MIRA_FW.getAuthorityManager().getPublishedChannels(id);
			foreach(const std::string& channel, publishedChannels)
			{
				boost::mutex::scoped_lock lock(mutex());
				ChannelNodePtr channelNode = getChannelNode(channel);
				mGraphHasChanged |= node->mPublishedChannels.insert(channelNode).second;
				mGraphHasChanged |= channelNode->mPublishingAuthorities.insert(node).second;
			}

			auto subscribedChannels = MIRA_FW.getAuthorityManager().getSubscribedChannels(id);
			foreach(const std::string& channel, subscribedChannels)
			{
				boost::mutex::scoped_lock lock(mutex());
				ChannelNodePtr channelNode = getChannelNode(channel);
				mGraphHasChanged |= node->mSubscribedChannels.insert(channelNode).second;
				mGraphHasChanged |= channelNode->mSubscribedAuthorities.insert(node).second;
			}

			graphHasChanged |= mGraphHasChanged;
			if(mGraphHasChanged && changeCb)
				changeCb();
		}
		return graphHasChanged;
	}

private:

	ChannelNodePtr getChannelNode(const std::string& id)
	{
		ChannelNodePtr node = findPtr(mChannels, id);
		if(!node) {
			node.reset(new ChannelNode(id));
			mChannels.push_back(node);
			NamespaceNodePtr ns = getNamespaceNode(ResourceName(id).parent());
			node->mNamespace = ns;
			ns->nodes.insert(node);
			mGraphHasChanged = true;
		}
		return node;
	}

	NamespaceNodePtr getNamespaceNode(const ResourceName& ns)
	{
		NamespaceNodePtr node = findPtr(mNamespaces, ns);
		if(!node) {
			node.reset(new NamespaceNode(ns));
			mNamespaces.push_back(node);
			if(!node->ns.isRoot())
				integrateNamespaceNode(node);
			mGraphHasChanged = true;
		}
		return node;
	}

	FrameworkNodePtr getFrameworkNode(const std::string& name)
	{
		FrameworkNodePtr node = findPtr(mFrameworks, name);
		if(!node) {
			node.reset(new FrameworkNode(name));
			mFrameworks.push_back(node);
			mGraphHasChanged = true;
		}
		return node;
	}

public:

	/// Returns all channel and authority nodes
	std::list<NodePtr> getNodes() {
		std::list<NodePtr> nodes;
		std::copy(getChannels().begin(),getChannels().end(),std::back_inserter(nodes));
		std::copy(getAuthorities().begin(),getAuthorities().end(),std::back_inserter(nodes));
		return nodes;
	}

	/// Returns all channel nodes
	std::list<ChannelNodePtr>& getChannels() {
		return mChannels; }

	/// Returns all authority nodes
	std::list<AuthorityNodePtr>& getAuthorities() {
		return mAuthorities; }

	/// Returns all namespace nodes
	std::list<NamespaceNodePtr>& getNamespaces() {
		return mNamespaces; }

	/// Returns all framework nodes
	std::list<FrameworkNodePtr>& getFrameworks() {
		return mFrameworks; }

	/// Returns the root namespace node
	NamespaceNodePtr getRootNamespace() {
		return mRootNamespace; }

	/// inserts a new namespace node and reparents the existing namespace nodes, if necessary.
	void integrateNamespaceNode(NamespaceNodePtr node)
	{
		assert(mRootNamespace);

		// find direct ancestor node
		NamespaceNodePtr ancestor = mRootNamespace;
		bool abort;
		do {
			abort = true;
			foreach(NamespaceNodePtr child, ancestor->children)
			{
				if(child->ns.isAncestorOf(node->ns)) {
					ancestor = child;
					abort = false;
					break;
				}
			}
		} while(!abort);


		// now add all children from ancestor that are actually our children
		// and remove them from ancestor
		auto childrenSet = ancestor->children;
		foreach(NamespaceNodePtr child, childrenSet)
		{
			if(child->ns.isChildOf(node->ns)) {
				node->children.insert(child);
				ancestor->children.erase(child);
				child->parent = node;
			}
		}

		// add us as new child to the ancestor
		ancestor->children.insert(node);
		// which is our parent
		node->parent = ancestor;

		// check if 'ancestor' is direct namespace parent of 'node'
		// if not, add all missing namespaces between 'ancestor' and 'node' recursively,
		// beginning with the direct parent of 'node'
		auto parent_ns = node->ns.parent();
		if (parent_ns != ancestor->ns)
		{
			NamespaceNodePtr n(new NamespaceNode(parent_ns));
			mNamespaces.push_back(n);
			integrateNamespaceNode(n);
		}
	}

	/*
	2nd version of the above method, work in progress
	void integrateNamespaceNode2(NamespaceNodePtr node)
	{
		assert(mRootNamespace);

		NamespaceNodePtr parent = getNamespaceNode(node->ns.parent());
		parent->children.insert(node);
		node->parent = parent;
	}
	*/

protected:

	boost::mutex mMutex;
	std::list<AuthorityNodePtr> mAuthorities;
	std::list<ChannelNodePtr> mChannels;
	std::list<NamespaceNodePtr> mNamespaces;
	std::list<FrameworkNodePtr> mFrameworks;
	NamespaceNodePtr mRootNamespace;

	bool mGraphHasChanged;



	template <typename Container, typename Key>
	typename Container::value_type findPtr(Container& c, const Key& key)
	{
		foreach(typename Container::value_type ptr, c)
			if(*ptr == key)
				return ptr;
		return typename Container::value_type();
	}
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
