/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelReadInterval.h
 *    Classes for reading whole intervals of data from channels.
 *
 * @author Erik Einhorn
 * @date   2010/09/17
 */

#ifndef _MIRA_CHANNELREADINTERVAL_H_
#define _MIRA_CHANNELREADINTERVAL_H_

#ifndef Q_MOC_RUN
#include <boost/shared_ptr.hpp>
#endif

#include <fw/ChannelReadWrite.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * An object that allows read access to a whole interval of channel data.
 * Use @ref Channel::readInterval to obtain access to data of a channel.
 *
 * The ChannelReadInterval contains a list of ChannelRead objects and
 * takes care of unlocking. One can iterate over the interval using the built
 * in iterators. It is also possible to take a single ChannelRead out of the
 * interval and to unlock the whole interval. The single ChannelRead will still
 * be read locked.
 */
template<typename T>
class ChannelReadInterval
{
	typedef ConcreteChannel<T>          ChannelType;
	typedef ChannelType*                    ChannelTypePtr;

	typedef typename ChannelType::Buffer    Buffer;
	typedef typename ChannelType::ValueType ValueType;
	typedef typename ChannelType::Slot      Slot;

public:
	typedef std::list<ChannelRead<T> > IntervalList;

	/// Const iterator for iterating over the interval
	class const_iterator
	{
	public:

		// some STL conform typedefs that are needed for iterators
		typedef ptrdiff_t                          difference_type;
		typedef std::bidirectional_iterator_tag    iterator_category;
		typedef ValueType                          value_type;
		typedef const ValueType*                   pointer;
		typedef const ValueType&                   reference;

	private:

		friend class ChannelReadInterval;
		const_iterator(const typename IntervalList::const_iterator& i) : mIterator(i) {}

	public:

		const_iterator(const const_iterator& other) : mIterator(other.mIterator) {}

	public:

		/// cast operator that casts to ChannelRead object
		operator ChannelRead<T>() const { return *mIterator; }

#if (__cplusplus >= 200710)
		/// cast operator that casts to original iterator
		explicit operator typename IntervalList::const_iterator() const { return mIterator; }
#else
		/// cast operator that casts to original iterator
		operator typename IntervalList::const_iterator() const { return mIterator; }
#endif

	public:

		reference operator*() const  { return (*(*mIterator)); }
		pointer   operator->() const { return &(*(*mIterator)); }

		const_iterator& operator++() {
			++mIterator;
			return *this;
		}

		const_iterator operator++(int) {
			iterator tmp = *this;
			++mIterator;
			return tmp;
		}

		const_iterator& operator--() {
			--mIterator;
			return *this;
		}

		const_iterator operator--(int) {
			iterator tmp = *this;
			--mIterator;
			return tmp;
		}

	public:

		bool operator==(const const_iterator& other) const {
			return mIterator==other.mIterator;
		}
		bool operator!=(const const_iterator& other) const {
			return mIterator!=other.mIterator;
		}

	private:

		typename IntervalList::const_iterator mIterator;
	};

	typedef const_iterator iterator;

public:

	const_iterator begin() const { return const_iterator(mInterval.begin()); }
	const_iterator end()   const { return const_iterator(mInterval.end());   }

public:

	std::size_t size() const { return mInterval.size(); }
	bool empty() const { return mInterval.empty(); }

public:

	/// Constructs an empty interval
	ChannelReadInterval() : mChannel(NULL) {}

	/// Constructs an interval on channel with given range
	ChannelReadInterval(ChannelTypePtr channel,
	                    std::list<ChannelBufferBase::Slot*>& slotList) :
		mChannel(channel)
	{
		// add all slots of the interval to our own interval list
		// create one ChannelRead object per slot within the interval
		for(auto it=slotList.begin(); it!=slotList.end(); ++it)
		{
			Slot* s = Buffer::castSlot(*it);

			// since the underlying channel buffer stores the slots starting
			// with the newest and ending with the oldest, we are moving
			// backward in time, however in our interval list we want to start
			// with the oldest element and end with the newest, so push_front
			// to revert the order.
			mInterval.push_front(ChannelRead<T>(channel, s));
		}
	}

public:

	void finish() {
		mInterval.clear();
		mChannel = NULL;
	}

	const std::string& getChannelID() {
		static std::string emptyString;
		return mChannel!=NULL ? mChannel->getID() : emptyString;
	}

	std::string getTypename() const {
		return mChannel!=NULL ? mChannel->getTypename() : "";
	}

	TypeMetaPtr getTypeMeta() const {
		return mChannel!=NULL ? mChannel->getTypeMeta() : TypeMetaPtr();
	}

	/**
	 * Returns a read-only channel proxy object of the underlying channel.
	 */
	Channel<T> getChannel();

private:

	void checkValid() const {
		if(mChannel==NULL)
			MIRA_THROW(XAccessViolation, "Trying to access ChannelReadInterval"
			           " that was not assigned with valid data");
	}

	ChannelTypePtr mChannel;
	IntervalList mInterval;
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
