/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AbstractChannel.h
 *    Base class for all framework channels.
 *
 * @author Erik Einhorn
 * @date   2010/09/08
 */

#ifndef _MIRA_ABSTRACTCHANNEL_H_
#define _MIRA_ABSTRACTCHANNEL_H_

#ifndef Q_MOC_RUN
#include <boost/thread/mutex.hpp>
#endif

#include <error/Exceptions.h>

#include <fw/AbstractChannelSubscriber.h>
#include <fw/ChannelBuffer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * @internal
 *
 * Abstract base class for all typed @ref Channel.
 * It allows to store different typed channels within one container.
 */
class MIRA_FRAMEWORK_EXPORT AbstractChannel : boost::noncopyable
{
public:

	typedef ChannelBufferBase::Slot Slot;

public:

	/// Constructs a new channel with the given id and the specified underlying channel buffer
	AbstractChannel(const std::string& id, ChannelBufferBase* buffer);

	/// Destructor (KEEP NON VIRTUAL!)
	~AbstractChannel();

public:

	/**
	 * Returns the channel id of this channel.
	 */
	const std::string& getID() const { return mID; }

public:

	/**
	 * Returns the non-portable typeid of the items that are stored in the
	 * slots of the underlying channel buffer.
	 * If the buffer is untyped this method will return -1.
	 * @return tpye id
	 */
	int getTypeId() const { return mBuffer->getTypeId(); }

	/**
	 * Returns the platform independent typename of the items that are stored
	 * in the slots of the underlying channel buffer. If the channel is untyped
	 * the typename that was set using setTypename() is returned.
	 * @return type name
	 */
	Typename getTypename() const { return mBuffer->getTypename(); }

	/**
	 * Sets the typename of the items that are stored in the slots of the
	 * underlying channel buffer. Calling this method is allowed for untyped
	 * channels only. Calling it for typed channels will result in a
	 * XRuntime exception.
	 * @throw XRuntime If the channel already is a typed channel.
	 * @param[in] name The new type name
	 */
	void setTypename(const Typename& name) { mBuffer->setTypename(name); }

	/**
	 * Returns the type meta information for the slots of the
	 * underlying channel buffer.
	 */
	TypeMetaPtr getTypeMeta() const { return mBuffer->getTypeMeta(); }

	/**
	 * Sets the type meta information for the slots of the
	 * underlying channel buffer.
	 */
	void setTypeMeta(TypeMetaPtr meta) { mBuffer->setTypeMeta(meta); }

	/**
	 * Calling this method will fix the type of the underlying ChannelBuffer.
	 * This method currently is used by the polymorphic buffer only. It will
	 * set the type of the buffer and hence the channel to the type that the
	 * buffer currently has, hence it fixates the type of the buffer and the
	 * buffer can not be promoted to any other type of a derived class.
	 */
	void fixateType() { mBuffer->fixateType(); }

	/**
	 *  Returns true, if the channel is typed and false, if it is untyped
	 */
	bool isTyped() const { return mBuffer->isTyped(); }

	/**
	 * Returns the underlying channel buffer. 
	 * Note that the buffer may change if the channel is promoted.
	 */
	ChannelBufferBase* getBuffer() { return mBuffer; }

	/**
	 * Returns the number of slots this channel contains.
	 */
	std::size_t getNrOfSlots() const { return mBuffer->getSize(); }

	/**
	 * Returns the timestamp of the data in the slot that was last written 
	 * to this channel or an invalid time when there was no data written at all.
	 */
	Time getLastSlotTime() const { return mLastSlotTime; }

	/**
	 * Returns the number of times data was written to this channel.
	 */
	uint64 getNrOfDataChanges() const { return mNrOfDataChanges; }

public:

	/**
	 * Returns true, if this channel has at least one subscriber
	 */
	bool hasSubscriber() const {
		return !mSubscribers.empty() || mNrOfSubscribersWithoutChannelSubscriber > 0; 
	}


	/**
	 * Returns true, if this channel has at least one publisher
	 */
	bool hasPublisher() const;

	/**
	 * Removes the specified subscriber from this channel and calls
	 * its detach() method.
	 *
	 * @note addSubscriber is implemented in ConcreteChannel
	 *       since type information is needed there.
	 */
	void removeSubscriber(AbstractChannelSubscriberPtr subscriber);

public:

	/**
	 * Is called by ChannelWrite to signal that data was changed
	 * @param[in] s Slot to be written into internal queue.
	 */
	void finishWrite(Slot* s);

	/**
	 * Is called by ChannelWrite to signal that this slot is discarded
	 * @param s Slot to be discarded
	 */
	void discardWrite(Slot* s);

public:

	/**
	 * Promote the channel to TargetType
	 */
	template<typename TargetType>
	void promote()
	{
		boost::mutex::scoped_lock lock(mPromoteMutex);
		ChannelBufferBase* oldBuffer = mBuffer;

		// promote our buffer, this may create a new buffer if necessary
		// then we set the new buffer and destroy the old one afterwards
		ChannelBufferBase* newBuffer;
		try {
			newBuffer = mBuffer->promote<TargetType>();
		}
		catch(Exception& ex) {
			MIRA_RETHROW(ex, "while trying to promote channel '" << this->getID() << "'")
		}

		if(newBuffer!=oldBuffer) {
			// TODO: the next assignment is atomic, hence there should be no
			// threading problem, even if someone obtains a ChannelRead or
			// ChannelWrite within the next line. (using a mutex here and in the
			// read() and write() methods would be even safer, but would also add
			// some more overhead.)
			mBuffer = newBuffer; // assign the new buffer

			// do not delete the old buffer here, as there might be
			// someone who still holds a lock to a slot. Instead, the buffer
			// will be freed when this AbstractChannel is destructed.
			// An alternative would be to use a shared_ptr for mBuffer and
			// to pass it to the ChannelReadWrite objects.
			mUntypedBuffer = oldBuffer;
		}
	}

public:

	/**
	 * Output the current buffer to console for debugging.
	 */
	void dbgDump(bool brief=true) { mBuffer->dbgDump(mID, brief); }

protected:

	std::string mID;
	ChannelBufferBase* mBuffer;
	ChannelBufferBase* mUntypedBuffer;

	boost::mutex mPromoteMutex;
	boost::mutex mSubscribersMutex;
	std::list<AbstractChannelSubscriberPtr > mSubscribers;
	uint32 mNrOfSubscribersWithoutChannelSubscriber; // number of subscribers that are subscribed without callback
	Time mLastSlotTime;
	uint64 mNrOfDataChanges;
};

/// Typedef for abstract channel pointers
typedef AbstractChannel* AbstractChannelPtr;

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

template<typename TargetType>
inline void promote_channel(AbstractChannelPtr channel)
{
	channel->promote<TargetType>();
}

template<>
inline void promote_channel<void>(AbstractChannelPtr channel)
{
	// Do nothing if target type is void (untyped channel). 
	// We can only promote to non-void typed channels.
}

///@endcond

///////////////////////////////////////////////////////////////////////////////

}

#endif
