/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLDomModifierTest.C
 *
 * @author Tim Langner, Christof Schröter
 * @date   2011/02/26
 */

#include <boost/test/unit_test.hpp>

#include <iostream>

#include <xml/XMLDomPreprocessor.h>

#include <utils/Foreach.h>
#include <utils/PackageFinder.h>

#include "CommonTest.h"

using namespace mira;
using namespace std;

XMLDom prepareDocuments(bool includeSpecialVariables)
{
	XMLDom xmlInclude;
	xmlInclude.root().
		add_child("IncludedNode").
		add_attribute("IntAttribute", 1).
		add_comment("This is a comment").
		add_content("This is content");
	xmlInclude.saveToFile("Include.xml");

	XMLDom xmlIncludeFrom;
	auto nodeOuter1 = 
		xmlIncludeFrom.root().
			add_child("NodeOuter1");

	nodeOuter1.
		add_child("NodeInner11").
		add_attribute("IntAttribute", 11).
		add_content("Content11");
	nodeOuter1.
		add_child("NodeInner12").
		add_attribute("IntAttribute", 12).
		add_content("Content12");
	nodeOuter1.
		add_child("NodeInner13").
		add_attribute("IntAttribute", 13).
		add_content("Content13");
	nodeOuter1.
		add_child("include").
		add_attribute("file", "IncludeFromInclude.xml").
		add_attribute("select", "NodeIncluded");

	auto nodeOuter2 =
		xmlIncludeFrom.root().
			add_child("NodeOuter2");

	nodeOuter2.
		add_child("NodeInner21").
			add_attribute("StrAttribute", "21").
			add_content("Content21");
	nodeOuter2.
		add_child("NodeInner22").
			add_attribute("StrAttribute", "22").
			add_content("Content22");

	auto nodeOuter3 =
		xmlIncludeFrom.root().
			add_child("NodeOuter3");
	auto nodeOuter4 =
		xmlIncludeFrom.root().
			add_child("NodeOuter4");

	nodeOuter3.
		add_child("NodeInner31").
			add_attribute("StrAttribute", "31").
			add_content("Content31");
	nodeOuter3.
		add_child("NodeInner32").
			add_attribute("StrAttribute", "32").
			add_content("Content32");

	auto nodeInner33 =
		nodeOuter3.
			add_child("NodeInner33");

	nodeInner33.
		add_child("NodeInnerInner331").
			add_attribute("StrAttribute", "331").
			add_content("Content331");
	nodeInner33.
		add_child("NodeInnerInner332").
			add_attribute("StrAttribute", "332").
			add_content("Content332");

	xmlIncludeFrom.saveToFile("IncludeFrom.xml");

	XMLDom xmlIncludeFromInclude;
	xmlIncludeFromInclude.root().
		add_child("NodeIncluded").
		add_content("Included");
	xmlIncludeFromInclude.root().
		add_child("NodeNotIncluded").
		add_content("NotIncluded");

	xmlIncludeFromInclude.saveToFile("IncludeFromInclude.xml");

	XMLDom xmlIncludeFrom2;
	xmlIncludeFrom2.root().
		add_child("NodeIncluded2a").
		add_content("Included");
	xmlIncludeFrom2.root().
		add_child("NodeIncluded2b").
		add_content("Included");
	xmlIncludeFrom2.root().
		add_child("NodeCritical").
		add_child("include").
		add_attribute("file", "NonExistentInclude.xml");
	xmlIncludeFrom2.saveToFile("IncludeFrom2.xml");

	XMLDom xml;
	xml.root().
		add_child("include").
		add_attribute("file", "Include.xml");
	xml.root().
		add_child("include").
		add_attribute("file", "IncludeFrom.xml").
		add_attribute("select", "NodeOuter1/*:NodeOuter2:NodeOuter4").
		add_attribute("select_not", "NodeOuter1:NodeOuter1/NodeInner12:NodeOuter4"); // 'NodeOuter1' will be ignored
	xml.root().
		add_child("include").
		add_attribute("file", "IncludeFrom.xml").
		add_attribute("select", "NodeOuter3").
		add_attribute("select_not", "NodeOuter3/NodeInner32:NodeOuter3/NodeInner33/*").
		add_attribute("as", "NodeAs");
	xml.root().
		add_child("include").
		add_attribute("file", "IncludeFrom2.xml").
		add_attribute("select_not", "NodeCritical");
	xml.root().
		add_child("DuplicateNode").
		add_child("include").
		add_attribute("file", "IncludeFromInclude.xml").
		add_attribute("select", "*:*");

	xml.root().
		add_child("if").
		add_attribute("color", "red").
		add_child("Red");
	xml.root().
		add_child("if").
		add_attribute("color", "green").
		add_child("Green");
	xml.root().
		add_child("if").
		add_attribute("color", "blue").
		add_child("include").
		add_attribute("file", "NonExistentInclude.xml");
	xml.root().
		add_child("if").
		add_attribute("name", "peter").
		add_child("Peter");
	xml.root().
		add_child("else").
		add_child("Heidi");
	if (includeSpecialVariables)
	{
		xml.root().
			add_child("RealName").
			add_attribute("name", "${name}");
		xml.root().
			add_child("RealColor").
			add_content("${color}");
	}
	return xml;
}

BOOST_AUTO_TEST_CASE( testInclude )
{
	XMLDom xml = prepareDocuments(false);
	XMLVariablesMap variables;
	preprocessXML(xml, variables);
	xml.saveToFile("ResolvedInclude.xml");
	BOOST_CHECK(xml.root().find("IncludedNode") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeOuter1") == xml.root().end());
	BOOST_CHECK(xml.root().find("NodeInner11") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeInner12") == xml.root().end());
	BOOST_CHECK(xml.root().find("NodeInner13") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeIncluded") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeOuter2/NodeInner21") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeOuter2/NodeInner22") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeOuter3") == xml.root().end());
	BOOST_CHECK(xml.root().find("NodeAs/NodeInner31") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeAs/NodeInner32") == xml.root().end());
	BOOST_CHECK(xml.root().find("NodeAs/NodeInner33") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeAs/NodeInner33/NodeInnerInner331") == xml.root().end());
	BOOST_CHECK(xml.root().find("NodeAs/NodeInner33/NodeInnerInner332") == xml.root().end());
	BOOST_CHECK(xml.root().find("NodeOuter4") == xml.root().end());
	BOOST_CHECK(xml.root().find("NodeNotIncluded") == xml.root().end());
	BOOST_CHECK(xml.root().find("NodeIncluded2a") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeIncluded2b") != xml.root().end());
	BOOST_CHECK(xml.root().find("NodeNotIncluded2") == xml.root().end());
	BOOST_CHECK(xml.root().find("DuplicateNode/NodeIncluded") != xml.root().end());
	BOOST_CHECK(xml.root().find("DuplicateNode/NodeIncluded", 1) != xml.root().end());
}

BOOST_AUTO_TEST_CASE( testIfReplace )
{
	XMLDom xml = prepareDocuments(true);
	XMLVariablesMap variables;
	variables["color"] = "red";
	variables["name"] = "peter";
	preprocessXML(xml, variables);
	xml.saveToFile("ResolvedIfTags.xml");
	BOOST_CHECK(xml.root().find("Red") != xml.root().end());
	BOOST_CHECK(xml.root().find("Peter") != xml.root().end());
	BOOST_CHECK(xml.root().find("RealName").get_attribute<std::string>("name") == "peter");
	BOOST_CHECK(*xml.root().find("RealColor").content_begin() == "red");
}

BOOST_AUTO_TEST_CASE( testInvalidConditions )
{
	XMLVariablesMap variables;

	std::list<std::string> cond_tags = {"if", "if_not", "if_all", "if_not_all", "if_any", "if_not_any",
	                                    "if_exists", "if_not_exists", "if_exists_all", "if_not_exists_all", "if_exists_any", "if_not_exists_any",
	                                    "assert", "assert_not", "assert_all", "assert_not_all", "assert_any", "assert_not_any",
	                                    "assert_exists", "assert_not_exists", "assert_exists_all", "assert_not_exists_all", "assert_exists_any", "assert_not_exists_any"};
	foreach (const std::string& t, cond_tags)
	{
		XMLDom xml;
		xml.root().
			add_child(t);
		BOOST_CHECK_THROW(preprocessXML(xml, variables), XInvalidConfig);
	}

	std::list<std::string> single_cond_tags = {"if", "if_not", "if_exists", "if_not_exists", "assert", "assert_not", "assert_exists", "assert_not_exists" };
	foreach (const std::string& t, single_cond_tags)
	{
		XMLDom xml;
		xml.root().
			add_child(t).
			add_attribute("cond1", "val1").
			add_attribute("cond2", "val2");
		BOOST_CHECK_THROW(preprocessXML(xml, variables), XInvalidConfig);
	}
}

BOOST_AUTO_TEST_CASE( testConditions )
{
	XMLVariablesMap variables;
	variables["true1"] = "true";
	variables["true2"] = "true";
	variables["false1"] = "false";
	variables["false2"] = "false";

	{
		XMLDom xml;
		xml.root().
			add_child("assert").
			add_attribute("true1", "true");
		preprocessXML(xml, variables);
	}
	{
		XMLDom xml;
		xml.root().
			add_child("assert_not").
			add_attribute("true1", "false");
		preprocessXML(xml, variables);
	}
	{
		XMLDom xml;
		xml.root().
			add_child("assert_all").
			add_attribute("true1", "true").
			add_attribute("true2", "true").
			add_attribute("false1", "false").
			add_attribute("false2", "false");
		preprocessXML(xml, variables);
	}
	{
		XMLDom xml;
		xml.root().
			add_child("assert_not_all").
			add_attribute("true1", "true").
			add_attribute("true2", "false");
		preprocessXML(xml, variables);
	}
	{
		XMLDom xml;
		xml.root().
			add_child("assert_not_all").
			add_attribute("true1", "false").
			add_attribute("true2", "true");
		preprocessXML(xml, variables);
	}
	{
		XMLDom xml;
		xml.root().
			add_child("assert_any").
			add_attribute("true1", "true").
			add_attribute("false1", "true");
		preprocessXML(xml, variables);
	}
	{
		XMLDom xml;
		xml.root().
			add_child("assert_any").
			add_attribute("true1", "false").
			add_attribute("false1", "false");
		preprocessXML(xml, variables);
	}
	{
		XMLDom xml;
		xml.root().
			add_child("assert_not_any").
			add_attribute("true1", "false").
			add_attribute("true2", "false").
			add_attribute("false1", "true").
			add_attribute("false2", "true");
		preprocessXML(xml, variables);
	}

	{
		XMLDom xml;
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("if").
			add_attribute("true1", "true").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "true");
	}
	{
		XMLDom xml;
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("if_not").
			add_attribute("true1", "true").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "false");
	}
	{
		XMLDom xml;
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("if_all").
			add_attribute("true1", "true").
			add_attribute("false1", "false").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "true");
	}
	{
		XMLDom xml;
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("if_not_all").
			add_attribute("true1", "false").
			add_attribute("false1", "false").
			add_child("var").
			add_attribute("result", "false").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "false");
	}
	{
		XMLDom xml;
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("if_any").
			add_attribute("true1", "false").
			add_attribute("false1", "false").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "true");
	}
	{
		XMLDom xml;
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("if_any").
			add_attribute("true1", "false").
			add_attribute("false1", "true").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "false");
	}
	{
		XMLDom xml;
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("if_not_any").
			add_attribute("true1", "false").
			add_attribute("true2", "false").
			add_attribute("false1", "true").
			add_attribute("false2", "true").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "true");
	}
	{
		XMLDom xml;
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("if_not_any").
			add_attribute("true1", "false").
			add_attribute("true2", "false").
			add_attribute("false1", "true").
			add_attribute("false2", "false").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false").
			add_attribute("overwrite", "true");
		xml.root().add_comment("this is a comment");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "false");
	}

}

BOOST_AUTO_TEST_CASE( testExistsMultipleVariables )
{
	{
		XMLVariablesMap variables;
		variables["myVar1"] = "myValue1";
		variables["myVar2"] = "myValue2";

		XMLDom xml;
		xml.root().
			add_child("if_exists_all").
			add_attribute("var1", "myVar1").
			add_attribute("variable2", "myVar2").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "true");
	}
	{
		XMLVariablesMap variables;
		variables["myVar1"] = "myValue1";
		variables["myVar2"] = "myValue2";

		XMLDom xml;
		xml.root().
			add_child("if_exists_all").
			add_attribute("var1", "myVar1").
			add_attribute("variable2", "myVar2").
			add_attribute("anothervariable", "invalid").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false");
		BOOST_CHECK_THROW(preprocessXML(xml, variables), XInvalidConfig);
	}
	{
		XMLVariablesMap variables;
		variables["myVar1"] = "myValue1";
		variables["myVar2"] = "myValue2";

		XMLDom xml;
		xml.root().
			add_child("if_exists_any").
			add_attribute("var_ABC", "doesNotExist").
			add_attribute("variable_whatever", "myVar1").
			add_child("var").
			add_attribute("result", "true").
			add_attribute("overwrite", "true");
		xml.root().
			add_child("else").
			add_child("var").
			add_attribute("result", "false");
		preprocessXML(xml, variables);
		BOOST_CHECK_EQUAL(variables["result"], "true");
	}
}

XMLDom getSubDom(XMLDom::const_sibling_iterator parent)
{
	XMLDom xml;
	for (auto node = parent.begin(); node != parent.end(); ++node)
		xml.root().add_child(node);

	return xml;
}

void testDeferResolve(bool explicitAttribute)
{
	XMLVariablesMap variables;
	variables["defer"] = "true";
	variables["color"] = "red";
	variables["name"] = "peter";

	XMLDom xmlInner = prepareDocuments(true);
	XMLDom xml;
	auto deferNode = xml.root().add_child("defer_resolve");
	if (explicitAttribute)
		deferNode.add_attribute("defer", "${defer}");

	for (auto node = xmlInner.root().begin(); node != xmlInner.root().end(); ++node)
		deferNode.add_child(node);

	// initially: content of deferNode = content of xmlInner's root
	BOOST_CHECK_EQUAL(xmlInner.saveToString(), getSubDom(deferNode).saveToString());
//	std::cout << xml.saveToString() << std::endl;

	// deferNode's children are not preprocessed
	// preprocessing adds or changes attribute on deferNode: "defer"="false"
	preprocessXML(xml, variables);
	BOOST_CHECK_EQUAL(xmlInner.saveToString(), getSubDom(deferNode).saveToString());
	BOOST_CHECK_EQUAL(deferNode.get_attribute<std::string>("defer"), "false");
//	std::cout << xml.saveToString() << std::endl;

	// deferNode's children are preprocessed now, result is equal to preprocessing xmlInnter
	// preprocessing keeps attribute on deferNode: "defer"="false"
	preprocessXML(xmlInner, variables);
	preprocessXML(xml, variables);
	BOOST_CHECK_EQUAL(xmlInner.saveToString(), getSubDom(deferNode).saveToString());
	BOOST_CHECK_EQUAL(deferNode.get_attribute<std::string>("defer"), "false");
//	std::cout << xml.saveToString() << std::endl;
}

BOOST_AUTO_TEST_CASE( testDeferResolveTag )
{
	testDeferResolve(false);
	testDeferResolve(true);

	XMLDom xml;
	auto deferNode = xml.root().add_child("defer_resolve").
	                            add_child("ThrowOnResolve").
	                            add_attribute("test", "${NonExistant}");

//	std::cout << xml.saveToString() << std::endl;

	XMLDomPreprocessor proc;

	// defers resolving first
	proc.preprocessXML(xml);
	// throws on trying to resolve now
	BOOST_CHECK_THROW(proc.preprocessXML(xml), XInvalidConfig);
}

std::pair<int, int> checkUriInfo(XMLDom::iterator& it)
{
	std::string name = *it;
	std::pair<int, int> counts(0, 0);

	std::string startsWith = name.substr(0, 1);

	if (startsWith == "A") {
		counts.first += 1;
		checkNodeUri(it, "A.xml", name.substr(1, std::string::npos));
	} else if (startsWith == "B") {
		counts.second += 1;
		checkNodeUri(it, "B.xml", name.substr(1, std::string::npos));
	}

	auto ch = it.begin();
	for (; ch != it.end(); ++ch) {
		std::pair<int, int> c = checkUriInfo(ch);
		counts.first += c.first;
		counts.second += c.second;
	}
	return counts;
}

BOOST_AUTO_TEST_CASE( testURI )
{
	XMLDom xml;
	Path dir = findPackage("MIRABase") / "tests" / "xml" / "etc";
	xml.root().add_child("include").add_attribute("file", (dir / "A.xml").string());
	auto rt = xml.root();

//	attributeUriInfo(rt);
//	std::cout << xml.saveToString() << std::endl;
//	std::cout << std::endl << std::endl;

	XMLDomPreprocessor proc;
	proc.preprocessXML(xml);

//	attributeUriInfo(rt);
//	std::cout << xml.saveToString() << std::endl;

	std::pair<int, int> counts = checkUriInfo(rt);
	BOOST_CHECK_EQUAL(counts.first, 5);
	BOOST_CHECK_EQUAL(counts.second, 5);
}

