/*
 * Copyright (C) 2015 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationVariantTest.C
 *    Tests for the variant serialization adapter
 *
 * @author Tom Mehner
 * @date   Mar 14, 2023
 */

#include <boost/test/unit_test.hpp>

#include "CommonTest.h"

#include <math/Angle.h>
#include <serialization/IgnoreMissing.h>
#include <serialization/adapters/boost/optional.hpp>
#include <serialization/adapters/boost/tuple.hpp>
#include <xml/XMLDomReflect.h>

#include <serialization/adapters/boost/variant.hpp>

// A class that deliberately does not reflect anything.
// Serialization/deserialization should work regardless.
struct Struct
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{}
};

class ComplexType
{
public:
	ComplexType() {}

	// set member to some values
	ComplexType(bool)
	{
		opt1.reset(123);
		opt2.reset();
		tup = boost::make_tuple(12, 3.45f, "678");
		guard = 0xABCD;
		phi = 1.25;
		xml.root().add_child("Node");
		variantMember = std::string("test string");
	}

	// returns true, if the values are as required
	void check()
	{
		BOOST_CHECK_EQUAL(*opt1, 123);
		BOOST_CHECK_EQUAL((bool)opt2, false);
		BOOST_CHECK_EQUAL(tup.get<0>(), 12);
		BOOST_CHECK_EQUAL(tup.get<1>(), 3.45f);
		BOOST_CHECK_EQUAL(tup.get<2>(), "678");
		BOOST_CHECK_EQUAL(guard, 0xABCD);
		BOOST_CHECK_EQUAL(phi.rad(), 1.25);
		BOOST_CHECK_EQUAL(*xml.root().begin(), "Node");
		BOOST_CHECK_EQUAL(boost::get<std::string>(variantMember), "test string");
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("opt1", opt1, "", serialization::IgnoreMissing());
		r.member("opt2", opt2, "", serialization::IgnoreMissing());
		r.member("tup", tup, "");
		r.member("guard", guard, "");
		r.member("phi", phi, "");
		r.member("xml", xml, "");
		r.member("struct", st, "");
		r.member("variantMember", variantMember, "");
	}

	boost::optional<int> opt1;
	boost::optional<int> opt2;
	boost::tuple<int, float, std::string> tup;
	int guard;
	Anglef phi;
	XMLDom xml;
	Struct st;

	boost::variant<int, std::string, bool> variantMember;
};

class ComplexerType
{
public:
	ComplexerType() {}

	// set member to some values
	ComplexerType(bool)
	{
		variantMember1 = ComplexType{true};
		variantMember2 = std::string("another test string");
		variantMember3 = true;
	}

	void check()
	{
		boost::get<ComplexType>(variantMember1).check();
		BOOST_CHECK_EQUAL(boost::get<std::string>(variantMember2), "another test string");
		BOOST_CHECK_EQUAL(boost::get<bool>(variantMember3), true);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("variantMember1", variantMember1, "");
		r.member("variantMember2", variantMember2, "");
		r.member("variantMember3", variantMember3, "");
	}

	boost::variant<ComplexType, std::string, bool> variantMember1;
	boost::variant<std::string, bool> variantMember2;
	boost::variant<std::string, ComplexType, bool> variantMember3;
};

BOOST_AUTO_TEST_CASE(VariantMemberTest)
{
	testAll<ComplexType>("VariantMemberClass", 1);
	testAll<ComplexerType>("VariantMemberClass2", 1);
}
