/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyTest.h
 *
 * @author Erik Einhorn
 */

#include <boost/noncopyable.hpp>
#include <boost/test/unit_test.hpp>

#include <serialization/PropertyManager.h>
#include <serialization/Serialization.h>
#include <serialization/SetterNotify.h>

#include <serialization/adapters/std/vector>

#include <utils/HasAssign.h>

#include <type_traits>

using namespace std;
using namespace mira;


class VeryBaseClass : public Object
{
	MIRA_OBJECT(VeryBaseClass)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
	}
};

class NonCopyableClassWithSubProperties : boost::noncopyable
{
public:

	NonCopyableClassWithSubProperties() : a(123), b(234) {}
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("a", a, "");
		r.property("b", b, "");
	}


	int a,b;
};

MIRA_CLASS_SERIALIZATION( VeryBaseClass, Object );

class BaseClass : public VeryBaseClass
{
	MIRA_OBJECT(BaseClass)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, VeryBaseClass);
		r.property("c", c, "this is c");
	}
	int c;
};

MIRA_CLASS_SERIALIZATION( BaseClass, VeryBaseClass );

class MyClass : public BaseClass
{
	MIRA_OBJECT(MyClass)
public:

	MyClass() : a(3), b(0.1234f) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, BaseClass);
		r.property("a", a, "this is a");
		r.property("b", b, "this is b");
		r.property("s", s, setterNotify(s, &MyClass::notify, this), "this is b");
	}

	void notify() {};

	int a;
	float b;
	std::string s;
};

MIRA_CLASS_SERIALIZATION( MyClass, BaseClass );

BOOST_AUTO_TEST_CASE( PropertyTest )
{
	PropertyManager m;
	vector<MyClass> v;
	v.resize(4);
	BaseClass* t = new MyClass();
	m.addObject("myobject",v);
	m.addObject("myclass",t);

	cout << "Property is: " << m.getProperty<int>("myobject.item[1].a") << endl;
	BOOST_CHECK_EQUAL(m.getProperty<int>("myobject.item[1].a"), 3);

	m.setProperty<int>("myobject.item[1].a", 4);
	cout << "Property is now: " << m.getProperty<int>("myobject.item[1].a") << endl;
	BOOST_CHECK_EQUAL(m.getProperty<int>("myobject.item[1].a"), 4);

	NonCopyableClassWithSubProperties noncopobj;
	m.addObject("noncopobj",noncopobj);

	cout << "Property is: " << m.getProperty<int>("noncopobj.a") << endl;
	m.setProperty<int>("noncopobj.a", 456);
	cout << "Property is now: " << m.getProperty<int>("noncopobj.a") << endl;
	BOOST_CHECK_EQUAL(m.getProperty<int>("noncopobj.a"), 456);

	PropertyNode* node = m.getPropertyNode("noncopobj");
	BOOST_CHECK(node!=NULL);
	std::cout << node->getAsString() << std::endl;
}
