/*
 * Copyright (C) 2015 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BigJSONTest.C
 *    Implementation of BigJSONTest.h.
 *
 * @author Tom Mehner
 */

#include <boost/test/unit_test.hpp>

#include <serialization/Serialization.h>
#include <serialization/adapters/std/shared_ptr.hpp>
#include <serialization/adapters/std/vector>

#include <math/Eigen.h>
#include <utils/ToString.h>

#include <chrono>
#include <memory>

using namespace std;
using namespace mira;

class Base : public mira::Object
{
	MIRA_OBJECT(Base)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{}
};
MIRA_CLASS_SERIALIZATION(Base, mira::Object)

class Derived : public Base
{
	MIRA_OBJECT(Derived)
public:
	Derived() {}
	Derived(int val) : a(val), v(10, 10)
	{
		v.setOnes(10, 10);
		v *= 1.0f * a;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
		r.member("v", v, "");
	}

	int a;
	Eigen::MatrixXf v;
};
MIRA_CLASS_SERIALIZATION(Derived, Base)

static const int N = 20000;

class BigJSONTestClass
{
public:
	BigJSONTestClass() {}

	explicit BigJSONTestClass(bool)
	{
		for (int i = 0; i < N; ++i)
			v.push_back(std::make_shared<Derived>(i));
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("v", v, "");
	}

	bool check()
	{
		BOOST_CHECK_EQUAL(v.size(), N);

		for (int i = 0; i < N; ++i)
			BOOST_CHECK(v[i]->getClass() == Derived::CLASS());

		for (int i = 0; i < N; ++i)
			BOOST_CHECK_EQUAL(((Derived*)v[i].get())->a, i);

		return true;
	}
	std::vector<std::shared_ptr<Base>> v;
};

BOOST_AUTO_TEST_CASE(BigXMLTest)
{
	using std::chrono::duration;
	using std::chrono::duration_cast;
	using std::chrono::high_resolution_clock;
	using std::chrono::milliseconds;

	std::cout << "Serializing ..." << std::endl;
	// serialize test class
	BigJSONTestClass c(true);

	auto serializationStart = high_resolution_clock::now();
	JSONSerializer serializer;
	json::Value v = serializer.serialize(c);
	auto serializationEnd = high_resolution_clock::now();

	auto durationSerialization = duration_cast<milliseconds>(serializationEnd - serializationStart);
	std::cout << "Serialization took: " << durationSerialization.count() << std::endl;

	std::cout << "Deserializing ..." << std::endl;
	// deserialize test class
	BigJSONTestClass c2; // with 0
	auto deserializationStart = high_resolution_clock::now();
	JSONDeserializer deserializer(v);
	deserializer.deserialize(c2);
	auto deserializationEnd = high_resolution_clock::now();

	auto durationDeserialization = duration_cast<milliseconds>(deserializationEnd - deserializationStart);
	std::cout << "Deserialization took: " << durationDeserialization.count() <<  std::endl;

	c2.check();
}
