/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RSAKeyTest.C
 *    Test case for RSAKey.
 *
 * @author Christian Martin
 * @date   2011/04/08
 */

#include <boost/test/unit_test.hpp>

#include <security/RSAKey.h>

#include <serialization/BinarySerializer.h>

#include <utils/ToString.h>

using namespace mira;
using namespace std;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RSAKeyTest_CopyCtor )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPrivateKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(tPrivateKey.isPrivateKey());

	RSAKey tPublicKeyCopy(tPublicKey);
	RSAKey tPrivateKeyCopy(tPrivateKey);

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_AssignOp )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPrivateKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(tPrivateKey.isPrivateKey());

	RSAKey tPublicKeyCopy;
	tPublicKeyCopy = tPublicKey;
	RSAKey tPrivateKeyCopy;
	tPrivateKeyCopy = tPrivateKey;

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_ToFromString )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPrivateKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(tPrivateKey.isPrivateKey());

	string tPublicStr = toString(tPublicKey);
	string tPrivateStr = toString(tPrivateKey);

	RSAKey tPublicKeyCopy = fromString<RSAKey>(tPublicStr);
	RSAKey tPrivateKeyCopy = fromString<RSAKey>(tPrivateStr);

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_N_E_D )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPrivateKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(tPrivateKey.isPrivateKey());

	string tPublicStrN = tPublicKey.getNStr();
	string tPublicStrE = tPublicKey.getEStr();
	string tPublicStrD = tPublicKey.getDStr();

	string tPrivateStrN = tPrivateKey.getNStr();
	string tPrivateStrE = tPrivateKey.getEStr();
	string tPrivateStrD = tPrivateKey.getDStr();

	RSAKey tPublicKeyCopy(tPublicStrN, tPublicStrE, tPublicStrD);
	RSAKey tPrivateKeyCopy(tPrivateStrN, tPrivateStrE, tPrivateStrD);

	BOOST_CHECK(tPublicKeyCopy.isValid());
	BOOST_CHECK(tPrivateKeyCopy.isValid());
	BOOST_CHECK(tPublicKeyCopy.isPublicKey());
	BOOST_CHECK(tPrivateKeyCopy.isPrivateKey());

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_clear )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	BOOST_CHECK(tPublicKey.isValid());
	BOOST_CHECK(tPublicKey.isPublicKey());
	BOOST_CHECK(!tPublicKey.isPrivateKey());

	tPublicKey.clear();
	BOOST_CHECK(!tPublicKey.isValid());
	BOOST_CHECK(!tPublicKey.isPublicKey());
	BOOST_CHECK(!tPublicKey.isPrivateKey());

	tPrivateKey.clear();
	BOOST_CHECK(!tPrivateKey.isValid());
	BOOST_CHECK(!tPrivateKey.isPublicKey());
	BOOST_CHECK(!tPrivateKey.isPrivateKey());
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_stream )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	stringstream tPubStream;
	tPubStream << tPublicKey;
	RSAKey tPublicKeyCopy;
	tPubStream >> tPublicKeyCopy;

	stringstream tPrivStream;
	tPrivStream << tPrivateKey;
	RSAKey tPrivateKeyCopy;
	tPrivStream >> tPrivateKeyCopy;

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}

BOOST_AUTO_TEST_CASE( RSAKeyTest_reflect )
{
	RSAKey tPublicKey, tPrivateKey;
	RSAKey::generateKey(1024, tPublicKey, tPrivateKey);

	Buffer<uint8> binaryBuffer;
	BinaryBufferSerializer binSerializer(&binaryBuffer);
	binSerializer.serialize(tPublicKey);
	binSerializer.serialize(tPrivateKey);

	RSAKey tPublicKeyCopy;
	RSAKey tPrivateKeyCopy;
	BinaryBufferDeserializer binDeserializer(&binaryBuffer);
	binDeserializer.deserialize(tPublicKeyCopy);
	binDeserializer.deserialize(tPrivateKeyCopy);

	BOOST_CHECK(tPublicKey == tPublicKeyCopy);
	BOOST_CHECK(tPrivateKey == tPrivateKeyCopy);
}
