/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RSAFilterTest.C
 *    Test case for RSAFilter.
 *
 * @author Christian Martin
 * @date   2016/06/17
 */

#include <boost/test/unit_test.hpp>

#include <boost/iostreams/device/file.hpp>
#include <boost/iostreams/device/file_descriptor.hpp>
#include <boost/iostreams/filtering_stream.hpp>

#include <boost/filesystem/operations.hpp>

#include <security/AESFilter.h>

#include <utils/ToString.h>

using namespace mira;
using namespace std;

///////////////////////////////////////////////////////////////////////////////

// Test all 3 bit length
AESBitLength sBitLengthList[] = { AES_128, AES_192, AES_256 };

// Test all 6 block cipher modes
AESBlockCipherMode sBCMList[] = { AES_ECB, AES_CBC, AES_CFB1, AES_CFB8, AES_CFB128, AES_OFB };

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( AESFilter_ShortDataTest )
{
	std::string key("Password");
	for(int m = 0; m < 3; m++) {
		for(int b = 0; b < 6; b++) {

			AESConfiguration cfg;
			cfg.bitLength = sBitLengthList[m];
			cfg.blockCipherMode = sBCMList[b];
			cfg.salt = "12345678";

			string tInputString = "Hello_world!";
			{
				boost::iostreams::filtering_ostream tOut;
				tOut.push(AESEncryptionFilter(cfg, key));
				tOut.push(boost::iostreams::file_sink("aes.dat", std::ios::out | std::ios::binary));
				tOut << tInputString << endl;
			}

			string tOutputString;
			{
				boost::iostreams::filtering_istream tIn;
				tIn.push(AESDecryptionFilter(cfg, key));
				tIn.push(boost::iostreams::file_source("aes.dat", std::ios::in | std::ios::binary));
				tIn >> tOutputString;
			}

			boost::filesystem::remove("aes.dat");

			BOOST_CHECK(tInputString == tOutputString);
		}
	}
}


BOOST_AUTO_TEST_CASE( AESFilter_MultiLineTest )
{
	std::string key("Password");
	for(int m = 0; m < 3; m++) {
		for(int b = 0; b < 6; b++) {

			AESConfiguration cfg;
			cfg.bitLength = sBitLengthList[m];
			cfg.blockCipherMode = sBCMList[b];
			cfg.salt = "12345678";

			string tInputLine1 = "Line1:_Hello_world!-Hello_world!-Hello_world!";
			string tInputLine2 = "Line2:_Hello_world!-Hello_world!-Hello_world!";
			string tInputLine3 = "Line3:_Hello_world!-Hello_world!-Hello_world!";
			{
				boost::iostreams::filtering_ostream tOut;
				tOut.push(AESEncryptionFilter(cfg, key));
				tOut.push(boost::iostreams::file_sink("aes.dat", std::ios::out | std::ios::binary));
				tOut << tInputLine1 << endl;
				tOut << tInputLine2 << endl;
				tOut << tInputLine3 << endl;
			}

			string tOutputLine1;
			string tOutputLine2;
			string tOutputLine3;
			{
				boost::iostreams::filtering_istream tIn;
				tIn.push(AESDecryptionFilter(cfg, key));
				tIn.push(boost::iostreams::file_source("aes.dat", std::ios::in | std::ios::binary));
				tIn >> tOutputLine1;
				tIn >> tOutputLine2;
				tIn >> tOutputLine3;
			}

			boost::filesystem::remove("aes.dat");

			BOOST_CHECK((tInputLine1 == tOutputLine1) &&
			            (tInputLine2 == tOutputLine2) &&
			            (tInputLine3 == tOutputLine3));
		}
	}
}


BOOST_AUTO_TEST_CASE( RSAFilter_LongDataTest )
{
	std::string key("Password");
	for(int m = 0; m < 3; m++) {
		for(int b = 0; b < 6; b++) {

			AESConfiguration cfg;
			cfg.bitLength = sBitLengthList[m];
			cfg.blockCipherMode = sBCMList[b];
			cfg.salt = "12345678";

			string tInputString1;
			for(int i = 0; i < 2000; i++)
				tInputString1 += "0123456789";
			string tInputString2;
			for(int i = 0; i < 1000; i++)
				tInputString2 += "0123456789";
			{
				boost::iostreams::filtering_ostream tOut;
				tOut.push(AESEncryptionFilter(cfg, key));
				tOut.push(boost::iostreams::file_sink("aes.dat", std::ios::out | std::ios::binary));
				tOut << tInputString1 << endl;
				tOut << tInputString2 << endl;
			}

			string tOutputString1;
			string tOutputString2;
			{
				boost::iostreams::filtering_istream tIn;
				tIn.push(AESDecryptionFilter(cfg, key));
				tIn.push(boost::iostreams::file_source("aes.dat", std::ios::in | std::ios::binary));
				tIn >> tOutputString1;
				tIn >> tOutputString2;
			}

			boost::filesystem::remove("aes.dat");

			BOOST_CHECK((tInputString1 == tOutputString1) &&
			            (tInputString2 == tOutputString2));
		}
	}
}
