/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RecursiveProcessTest.C
 *
 * @author Christof Schröter
 * @date   2021/02/05
 */

#include <fstream>
#include <sys/stat.h>

#ifdef BOOST_TEST_MODULE
#include <boost/test/unit_test.hpp>
#endif

#include <platform/Process.h>
#include <error/LogConsoleSink.h>
#include <thread/Thread.h>
#include <utils/Path.h>

#include <proc/readproc.h>

using namespace mira;

void createScripts(const Path& tmp, const std::string& command, bool silent, int scriptLevels = 2)
{
	for (int l = 0; l < scriptLevels; ++ l) {
		Path file = tmp / ("recurse"+toString(l)+".sh");
		Path fileNext = tmp / ("recurse"+toString(l+1)+".sh");
		{
			std::ofstream s(file.string(), std::ofstream::trunc);
			if (l == scriptLevels-1) {
				s << command << " -$1 0 " << silent << std::endl; // negate level to signal the process it is not topLevel
				if (!silent)
					s << "echo \"recurse" << toString(l) << ".sh level $1 done\"" << std::endl;
			} else {
				s << fileNext.string() << " $1" << std::endl;
				if (!silent)
					s << "echo \"recurse" << toString(l) << ".sh level $1 done\"" << std::endl;
			}
		}
		chmod(file.c_str(), S_IRWXU|S_IRWXG|S_IRWXO); // r+w+x for u,o,g (="everyone may do everything")
	}
}

void run(const std::string& command, int depth, bool topLevel, bool shutdownRecursively, bool silent)
{
	Time shutdownStartTime;

	if (depth > 0) {
		Process p = Process::createProcess(command,
		                                   {std::to_string(depth-1)},
		                                   shutdownRecursively ? Process::CreationFlags::shutdownRecursively
		                                                       : Process::CreationFlags::noflags,
		                                   Process::none);

		if (topLevel) {
			std::cout << "Level " << depth << " waiting for 3s..." << std::endl;
			MIRA_SLEEP(3000);
			std::cout << "Level " << depth
			          << (shutdownRecursively ? " stopping + all children!" : " stopping.") << std::endl;
			shutdownStartTime = Time::now();
		} else {
			if (!silent)
				std::cout << "Level " << depth << " waiting for child process..." << std::endl;
			p.wait();
			if (!silent)
				std::cout << "Level " << depth << " stopping." << std::endl;
		}
	} else {
		for (int n = 0; n < 20; ++n) {
			if (!silent)
				std::cout << "== Level " << depth << " alive ==" << std::endl;
			MIRA_SLEEP(3000);
		}
		if (!silent)
			std::cout << "== Level " << depth << " out ==" << std::endl;
	}

	if (shutdownStartTime.isValid())
		std::cout << "Stopping process (hierarchy) took " << (Time::now() - shutdownStartTime).totalSeconds() << "s" << std::endl;
}

#ifndef BOOST_TEST_MODULE
int main(int argc, char** argv)
{
	if (argc < 3) {
		std::cout << "Usage: RecursiveProcessTestBinary recurseDepth shutdownRecursively [silent] (e.g. RecursiveProcessTest 3 1 1)" << std::endl;
		exit(-1);
	}

	int depth = std::stoi(argv[1]);
	bool topLevel = depth > 0;
	depth = std::abs(depth);
	std::string s2(argv[2]);
	bool shutdownRecursively = (s2 == "true" || s2 == "1");

	bool silent = false;
	if (argc >= 4) {
		std::string s3(argv[3]);
		silent = (s3 == "true" || s3 == "1");
	}

	Path tmp = getTempDirectory();
	if (topLevel)
		createScripts(tmp, argv[0], silent);

	std::string command = (tmp / "recurse0.sh").string();
	run(command , depth, topLevel, shutdownRecursively, silent);

	if (topLevel) {
		if (shutdownRecursively)
			std::cout << "All created processes have been (forcibly) stopped." << std::endl;
		else
			std::cout << "Parts of the created process hierarchy are still alive! They will run out after ~1min" << std::endl;
	}
}
#else
std::vector<std::string> getProcessList()
{
	PROCTAB* tab = NULL;
	int count = 0;
	while (!tab) {
		tab = ::openproc(PROC_FILLCOM);
		MIRA_SLEEP(10);
		++count;
	}

	std::vector<std::string> processes;

	for (;;) {
		proc_t* proc = ::readproc(tab, NULL);
		if (!proc)
			break;

		if (proc->cmdline)
			processes.emplace_back(proc->cmdline[0]);

		freeproc(proc);
	}

	closeproc(tab);
	return processes;
}

BOOST_AUTO_TEST_CASE( TestNonRecursiveShutdown )
{
	Path test = boost::unit_test::framework::master_test_suite().argv[0];
	std::string binary = (test.parent_path()/"RecursiveProcessTestBinary").string();
	Path tmp = getTempDirectory();
	
	createScripts(tmp, binary, true, 1);

	std::string command = (tmp / "recurse0.sh").string();
	run(command, 2, true, false, true);

	// check something has survived
	auto processes = getProcessList();
	BOOST_CHECK(std::find(processes.begin(), processes.end(), binary) != processes.end());

	// give created processes enough time to stop, but make sure the test finishes
	for (int n = 0; n < 20; ++n) {
		MIRA_SLEEP(5000);
		processes = getProcessList();
		if (std::find(processes.begin(), processes.end(), binary) == processes.end())
			return;
	}

	BOOST_FAIL("Created processes never stopped");
}

BOOST_AUTO_TEST_CASE( TestRecursiveShutdown )
{
	Path test = boost::unit_test::framework::master_test_suite().argv[0];
	std::string binary = (test.parent_path()/"RecursiveProcessTestBinary").string();
	Path tmp = getTempDirectory();
	
	createScripts(tmp, binary, true, 1);

	std::string command = (tmp / "recurse0.sh").string();
	run(command, 2, true, true, true);

	// check nothing has survived
	auto processes = getProcessList();
	BOOST_CHECK(std::find(processes.begin(), processes.end(), binary) == processes.end());
}
#endif
