/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RasterTriangleTest.C
 *    A test case for triangle rasterization.
 *
 * @author Erik Einhorn
 * @date   2010/12/06
 */

#include <boost/test/unit_test.hpp>

#include <geometry/RasterTriangle.h>
#include <iostream>

using namespace std;
using namespace mira;

///////////////////////////////////////////////////////////////////////////////

struct TriangleElement
{
	int x, y, length;
};

///////////////////////////////////////////////////////////////////////////////

struct TriangleGroundTruth
{
	int x0, y0, x1, y1, x2, y2;
	int groundTruthSize;
	TriangleElement groundTruth[32];
};

///////////////////////////////////////////////////////////////////////////////

TriangleGroundTruth triangleTests[] = {
	////////////////////
	{2,0, 0,6, 7,7,
		8, {
			{2,0, 1},
			{2,1, 2},
			{1,2, 3},
			{1,3, 4},
			{1,4, 5},
			{0,5, 7},
			{0,6, 7},
			{4,7, 4},
			}
	},
	{2,0, 4,0, 6,2,  // degenerate case 1a: y0 = y1 (x0 < x1)
		3, {
			{2,0, 3},
			{3,1, 3},
			{5,2, 2}
			}
	},
	{4,0, 2,0, 6,2,  // degenerate case 1b: y0 = y1 (x0 > x1)
		3, {
			{2,0, 3},
			{3,1, 3},
			{5,2, 2}
			}
	},
	{2,4, 4,4, 6,2,  // degenerate case 2a: y1 = y2 (x1 < x2)
		3, {
			{6,2, 1}, // not symmetrical to 1a, because the line (triangle edge)
			{4,3, 2}, // rastering varies slightly for non integral x/y ratios
			{2,4, 3}
			}
	},
	{4,4, 2,4, 6,2,  // degenerate case 2b: y1 = y2 (x1 > x2)
		3, {
			{6,2, 1},
			{4,3, 2},
			{2,4, 3}
			}
	},
	{2,4, 4,4, 6,4,  // degenerate case 3a: y0 = y1 = y2 (straight line x-aligned), x sorted
		1, {
			{2,4, 5}
			}
	},
	{4,4, 2,4, 6,4,  // degenerate case 3b: y0 = y1 = y2 (straight line x-aligned), x1 < x0
		1, {
			{2,4, 5}
			}
	},
	{4,4, 6,4, 2,4,  // degenerate case 3c: y0 = y1 = y2 (straight line x-aligned), x2 < x1
		1, {
			{2,4, 5}
			}
	},
	{4,4, 4,2, 4,6,  // degenerate case 4: x0 = x1 = x2 (straight line y-aligned)
		5, {
			{4,2, 1},
			{4,3, 1},
			{4,4, 1},
			{4,5, 1},
			{4,6, 1}
			}
	},
	{1,1, 4,4, 3,3,  // degenerate case 5a: straight line, not x/y-aligned, x1 > x0
		4, {
			{1,1, 1},
			{2,2, 1},
			{3,3, 1},
			{4,4, 1}
			}
	},
	{4,4, 1,1, 3,3,  // degenerate case 5b: straight line, not x/y-aligned, x1 < x0
		4, {
			{1,1, 1},
			{2,2, 1},
			{3,3, 1},
			{4,4, 1}
			}
	},
	{1,1, 1,1, 4,3,  // degenerate case 6aaa: x0 = x1, y0 = y1, x2 > x0/1, y2 != y0/1 (p0/1 identical, line not x/y-aligned, to right)
		3, {
			{1,1, 1},
			{2,2, 2},
			{4,3, 1}
			}
	},
	{4,1, 4,1, 1,3,  // degenerate case 6aab: x0 = x1, y0 = y1, x2 < x0/1, y2 != y0/1 (p0/1 identical, line not x/y-aligned, to left)
		3, {
			{4,1, 1},
			{2,2, 2},
			{1,3, 1}
			}
	},
	{1,1, 1,1, 4,1,  // degenerate case 6aba: x0 = x1, y0 = y1 = y2, x2 > x0/1 (p0/1 identical, line x-aligned, to right)
		1, {         // (special case of 3a)
			{1,1, 4}
			}
	},
	{4,1, 4,1, 1,1,  // degenerate case 6abb: x0 = x1, y0 = y1 = y2, x2 < x0/1 (p0/1 identical, line x-aligned, to left)
		1, {         // (special case of 3b)
			{1,1, 4}
			}
	},
	{1,1, 1,1, 1,4,  // degenerate case 6ac: x0 = x1 = x2, y0 = y1, y2 != y0/1 (p0/1 identical, line y-aligned)
		4, {
			{1,1, 1},
			{1,2, 1},
			{1,3, 1},
			{1,4, 1}
			}
	},
	{1,1, 4,3, 4,3,  // degenerate case 6baa: x1 = x2, y1 = y2, x0 < x1/2, y0 != y1/2 (p1/2 identical, line not x/y-aligned, to right)
		3, {
			{1,1, 1},
			{2,2, 2},
			{4,3, 1}
			}
	},
	{4,1, 1,3, 1,3,  // degenerate case 6bab: x1 = x2, y1 = y2, x0 > x1/2, y0 != y1/2 (p1/2 identical, line not x/y-aligned, to left)
		3, {
			{4,1, 1},
			{2,2, 2},
			{1,3, 1}
			}
	},
	{1,1, 4,1, 4,1,  // degenerate case 6bba: x1 = x2, y0 = y1 = y2, x0 < x1/2 (p1/2 identical, line x-aligned, to right)
		1, {         // (special case of 3a)
			{1,1, 4}
			}
	},
	{4,1, 1,1, 1,1,  // degenerate case 6bbb: x1 = x2, y0 = y1 = y2, x0 > x1/2 (p1/2 identical, line x-aligned, to left)
		1, {         // (special case of 3b)
			{1,1, 4}
			}
	},
	{1,1, 1,4, 1,4,  // degenerate case 6bc: x0 = x1 = x2, y1 = y2, y0 != y1/2 (p1/2 identical, line y-aligned)
		4, {
			{1,1, 1},
			{1,2, 1},
			{1,3, 1},
			{1,4, 1}
			}
	},
	{1,4, 1,4, 1,4,  // degenerate case 7: x0 = x1 = x2, y0 = y1 = y2 (p0/1/2 identical)
			1, {
				{1,4, 1}
			}
	},

	// more test cases from Tim van der Grinten (CASE designation might not resemble current code, but tests are still valuable)
	// CASE 0
	{0,0, 0,8, 0,5,
		9, {
			{0,0, 1},
			{0,1, 1},
			{0,2, 1},
			{0,3, 1},
			{0,4, 1},
			{0,5, 1},
			{0,6, 1},
			{0,7, 1},
			{0,8, 1},
			}
	},
	{0,0, 2,6, 1,3,
		7, {
			{0,0, 1},
			{0,1, 1},
			{1,2, 1},
			{1,3, 1},
			{1,4, 1},
			{2,5, 1},
			{2,6, 1},
			}
	},
	// CASE 1
	{0,0, 0,2, 1,1,
		3, {
			{0,0, 1},
			{0,1, 2},
			{0,2, 1},
			}
	},
	{-1,0, 1,0, 0,1,
		2, {
			{-1,0, 3},
			{ 0,1, 1},
			}
	},
	{-2,0, 0,6, -7,7,
		8, {
			{-2,0, 1},
			{-3,1, 2},
			{-3,2, 3},
			{-4,3, 4},
			{-5,4, 5},
			{-6,5, 7},
			{-6,6, 7},
			{-7,7, 4},
			}
	},
	// CASE 2
	{0,0, 0,2, -1,1,
		3, {
			{0,0, 1},
			{-1,1, 2},
			{0,2, 1},
			}
	},
	{1,0, -1,0, 0,1,
		2, {
			{-1,0, 3},
			{ 0,1, 1},
			}
	},
	{2,0, 0,0, -2,0,
		1, {
			{-2,0, 5},
			}
	},
	// VERTICAL TEST
	{0,0, 1,5, 1,3,
		6, {
			{0,0, 1},
			{0,1, 1},
			{0,2, 2},
			{1,3, 1},
			{1,4, 1},
			{1,5, 1},
			}
	},
	{0,0, 1,5, 0,2,
		6, {
			{0,0, 1},
			{0,1, 1},
			{0,2, 1},
			{0,3, 2},
			{1,4, 1},
			{1,5, 1},
			}
	},
	// HORIZONTAL TEST
	{0,0, 5,1, 2,0,
		2, {
			{0,0, 4},
			{3,1, 3},
			}
	},
	{0,0, 5,1, 3,1,
		2, {
			{0,0, 3},
			{3,1, 3},
			}
	},
	{2,0, 0,0, 5,1,
		2, {
			{0,0, 4},
			{3,1, 3},
			}
	},
	{0,0, 3,1, 5,1,
		2, {
			{0,0, 3},
			{3,1, 3},
			}
	},
};

///////////////////////////////////////////////////////////////////////////////

class TriangleTester
{
public:

	TriangleTester(const TriangleGroundTruth& it) : t(it), i(0)
	{
		passed = true;
	}

	void operator()(int xl, int xr, int y)
	{
		int l =  (xr-xl+1);
		cout << "xl: " << xl << ",y: " << y << " l: " << l << endl;

		if(i>=t.groundTruthSize) {
			passed = false;
			cout << "too many segments! current segment: ("
			     << xl << ", " << y << ", " << l << ")" << endl;
		}

		const TriangleElement& e = t.groundTruth[i];
		if(e.x!=xl || e.y!=y || e.length!=l) {
			passed = false;
			cout << "wrong segment! is: ("
			     << xl << ", " << y << ", " << l
			     << "), should: ("
			     << e.x << ", " << e.y << ", " << e.length << ")" << endl;
		}
		++i;
	}

	const TriangleGroundTruth& t;
	int i;
	bool passed;
};

///////////////////////////////////////////////////////////////////////////////

void testTriangle(const TriangleGroundTruth& t)
{
	TriangleTester tester(t);
	rasterTriangle(Point2i(t.x0,t.y0), Point2i(t.x1,t.y1), Point2i(t.x2,t.y2),
	               tester);
	BOOST_CHECK(tester.passed);
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( VisitorTest )
{
	for(size_t i = 0; i<sizeof(triangleTests)/sizeof(TriangleGroundTruth); ++i)
		testTriangle(triangleTests[i]);
}
