/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Time.C
 *    Implementation of Time.h
 *
 * @author Christian Martin
 * @date   2011/06/28
 */

#include <utils/Time.h>

#include <boost/optional.hpp>
#include <boost/thread.hpp>

#include <error/LoggingCore.h>
#include <thread/Thread.h>
#include <utils/Singleton.h>

#ifdef MIRA_WINDOWS
# include <boost/chrono.hpp>
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

#ifdef MIRA_WINDOWS

class TimeSynchronizer : public LazySingleton<TimeSynchronizer>
{
public:
	TimeSynchronizer() {
		mSyncThread = boost::thread(boost::bind(&TimeSynchronizer::processSync, this));
	}
	
	~TimeSynchronizer() {
		mSyncThread.interrupt();
		mSyncThread.join();
	}

	void processSync()
	{
		while (!mSyncThread.interruption_requested()) {
			// Get the current system time before the loop.
			auto systemClockT0 = boost::chrono::system_clock::now();
	
			// Spin until the system time changes
			boost::chrono::system_clock::time_point systemClockT1;
			boost::chrono::high_resolution_clock::time_point highresClockT1;
			int maxLoopCnt = 25000;
			do {
				systemClockT1 = boost::chrono::system_clock::now();
				highresClockT1 = boost::chrono::high_resolution_clock::now();
				maxLoopCnt--;
			} while ((systemClockT1 == systemClockT0) && (maxLoopCnt > 0));

			// If we didn't found a sync time, try it later.
			if (maxLoopCnt == 0) {
				if (!mBootupTime)
					MIRA_LOG(WARNING) << "Unable to synchronize high resolution clock. Trying again.";
				MIRA_SLEEP(2000);
				continue;
			}
		
			// At this point, we have a new point in time, where we can
			// synchronize the system time (systemClockT1) and the high
			// resolution timer (highresClockT1).
		
			// Convert the system time into a unix time in [s]
			uint64 t0 = boost::chrono::system_clock::to_time_t(systemClockT1);

			// Compute the rounding error of the conversion above.
			auto rounded = boost::chrono::system_clock::from_time_t(t0);
			int32 fraction = boost::chrono::duration_cast<boost::chrono::microseconds>(systemClockT1 - rounded).count();

			// Time since bootup based on the high resolution timer in [ns]
			uint64 t1 = highresClockT1.time_since_epoch().count();
			
			// Compute the bootup time
			Time bootupTime = Time::fromUnixNS(t0*1000000000-t1)+Duration::microseconds(fraction);
			mBootupTime.reset(bootupTime);
			
			// Sleep a little bit
			MIRA_SLEEP(2000);
		}
	}

public:
	boost::optional<Time> mBootupTime;
	
private:
	boost::thread mSyncThread;
};

///////////////////////////////////////////////////////////////////////////////

Time Time::now()
{
	TimeSynchronizer::instance();

	static bool sFirstCall = true;
	if (sFirstCall) {
		sFirstCall = false;

		// Sleep here for 2s to increase the chance, that we are able
		// to synchronize the high resolution timer as soon as possible.
		MIRA_SLEEP(2000);
	}

	if (!TimeSynchronizer::instance().mBootupTime) {
		// As long we don't have a valid sync point, only return the system
		// time, which is not very accurate (depending on the Windows version).
		return boost::posix_time::microsec_clock::universal_time();
	} else {
		// Determine the time based on the last know synchronization point of
		// the system time and the high resolution timer.
		return *(TimeSynchronizer::instance().mBootupTime) +
			Duration::nanoseconds(boost::chrono::high_resolution_clock::now().time_since_epoch().count());
	}
}

#endif

///////////////////////////////////////////////////////////////////////////////

}
