/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SHA256.C
 *    Implementation of SHA256.h
 *
 * @author Christian Martin
 * @date   2010/09/10
 */

#include <security/SHA256.h>

#include <openssl/sha.h>

#include <error/Exceptions.h>

#include "OpenSSLHelper.h"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

struct SHA256StreamBuf::Context
{
	SHA256_CTX sha_ctx;
};

///////////////////////////////////////////////////////////////////////////////

SHA256StreamBuf::SHA256StreamBuf() :
	mCtx(NULL)
{
	OpenSSLCleanup::instance();

	mCtx = new Context;
	memset(mCtx, 0x00, sizeof(Context));

	reset();
}

SHA256StreamBuf::~SHA256StreamBuf()
{
	delete mCtx;
	mCtx = NULL;
}

void SHA256StreamBuf::reset()
{
	if (SHA256_Init(&mCtx->sha_ctx) != 1)
		MIRA_THROW(XSystemCall, "SHA256_Init failed.");
}

HashDigest SHA256StreamBuf::getDigest() const
{
	HashDigest tDigest(SHA256_DIGEST_LENGTH);
	if (SHA256_Final((unsigned char*)&tDigest.at(0), &mCtx->sha_ctx) != 1)
		MIRA_THROW(XSystemCall, "SHA256_Final failed.");
	return(tDigest);
}

std::streamsize SHA256StreamBuf::xsputn(const char* s, std::streamsize n)
{
	if (SHA256_Update(&mCtx->sha_ctx, s, (size_t)n) != 1)
		MIRA_THROW(XSystemCall, "SHA256_Update failed.");
	return(n);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace
