/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JSON.C
 *    Implementation of JSON.h
 *
 * @author Erik Einhorn
 * @date   2011/12/20
 */

#include <json_spirit_utils.h>
#include <json_spirit_writer_template.h>
#include <json_spirit_reader_template.h>

#include <utils/StringAlgorithms.h>
#include <math/EigenFormat.h>

#include <json/JSON.h>

#ifndef MIRA_WINDOWS
template class json_spirit::Config_map<std::string>;
template class json_spirit::Value_impl<json_spirit::Config_map<std::string>>;
#endif

namespace mira { namespace json {

///////////////////////////////////////////////////////////////////////////////

void write(const Value& value, std::ostream& ioStream,
           bool formatted, int precision)
{
	int options = json_spirit::remove_trailing_zeros; // deprecated, no effect!
	if (formatted)
		// single_line_arrays implies pretty_print for json spirit currently, but better not rely on it
		options |= json_spirit::pretty_print | json_spirit::single_line_arrays;
	json_spirit::write_stream(value, ioStream, options,
	                          precision < 0 ? JSONDefaultPrecision::get() : precision);
}

std::string write(const Value& value, bool formatted, int precision)
{
	int options = json_spirit::remove_trailing_zeros; // deprecated, no effect!
	if (formatted)
		// single_line_arrays implies pretty_print for json spirit currently, but better not rely on it
		options |= json_spirit::pretty_print | json_spirit::single_line_arrays;
	return json_spirit::write_string(value, options,
	                                 precision < 0 ? JSONDefaultPrecision::get() : precision);
}

void read(const std::string& s, Value& oValue)
{
	try
	{
		json_spirit::read_string_or_throw(s, oValue);
	}
	catch(std::string& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string '" << s << "': " << ex);
	}
	catch(json_spirit::Error_position& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string '" << s << "': On line "
		           << ex.line_ << " column " << ex.column_ << " reason " << ex.reason_);
	}
	catch(...)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string '" << s << "'");
	}
}

void read(std::istream& ioStream, Value& oValue)
{
	try
	{
		json_spirit::read_stream_or_throw(ioStream, oValue);
	}
	catch(std::string& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: " << ex);
	}
	catch(json_spirit::Error_position& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: On line "
		           << ex.line_ << " column " << ex.column_ << " reason " << ex.reason_);
	}
	catch(...)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string");
	}
}

void read(std::string::const_iterator& begin,
          std::string::const_iterator& end, Value& oValue)
{
	try
	{
		json_spirit::read_range_or_throw(begin, end, oValue);
	}
	catch(std::string& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: " << ex);
	}
	catch(json_spirit::Error_position& ex)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string: On line "
		           << ex.line_ << " column " << ex.column_ << " reason " << ex.reason_);
	}
	catch(...)
	{
		MIRA_THROW(XIO, "Error reading JSON value from string");
	}
}

std::pair<const Value*, std::string> accessElement(const Value& iValue, const std::string& element)
{
	const Value* currentValue = &iValue;
	if (!element.empty())
	{
		std::vector<std::string> out;
		boost::split(out, element, boost::is_from_range('.','.'));
		for(std::size_t i=0; i<out.size(); ++i)
		{
			// check for array access
			std::size_t bBegin = out[i].find("[");
			if (bBegin != std::string::npos)
			{
				std::size_t bEnd = out[i].find("]", bBegin);
				if (bEnd == std::string::npos || (int)bEnd - (int)bBegin <= 1)
					return std::pair<const Value*, std::string>(
						NULL, "json::accessElement(): Wrong syntax for json::Array element "
						      "(check position and number of brackets, use [index])");
				if (bBegin > 0)
				{
					if (currentValue->type() != json_spirit::obj_type)
						return std::pair<const Value*, std::string>(
							NULL, "json::accessElement(): Data must be a json::Object if member is specified");
					std::string objName = out[i].substr(0, bBegin);
					if (currentValue->get_obj().find(objName) == currentValue->get_obj().end())
						return std::pair<const Value*, std::string>(
							NULL, MakeString() << "json::accessElement(): Member element '" << element << "' not found");
					currentValue = &currentValue->get_obj().at(objName);
				}
				std::size_t index = fromString<std::size_t>(out[i].substr(bBegin+1, 
				                                                          bEnd-bBegin-1));
				if (currentValue->type() == json_spirit::array_type)
				{
					if (index >= currentValue->get_array().size())
						return std::pair<const Value*, std::string>(
							NULL, "json::accessElement(): Array element index is out of range");
					currentValue = &currentValue->get_array()[index];

					// check for valid multidim index syntax and give a helpful error message if necessary
					if (out[i].find("[", bEnd) != std::string::npos)
						return std::pair<const Value*, std::string>(
							NULL, "json::accessElement(): Wrong syntax for nested json::Array "
							      "(use [i].[j] for multiple dimension indices)");
				}
				else
					return std::pair<const Value*, std::string>(
						NULL, "json::accessElement(): Data must be a json::Array if index is specified");
				continue;
			}

			if (currentValue->type() != json_spirit::obj_type)
				return std::pair<const Value*, std::string>(
					NULL, "json::accessElement(): Data must be a json::Object if member is specified");
			if (currentValue->get_obj().find(out[i]) == currentValue->get_obj().end())
				return std::pair<const Value*, std::string>(
					NULL, MakeString() << "json::accessElement(): Member element '" << element << "' not found");
			currentValue = &currentValue->get_obj().at(out[i]);
		}
	}
	return std::make_pair(currentValue, "");
}

Value getElement(const Value& iValue, const std::string& element)
{
	std::pair<const Value*, std::string> v = accessElement(iValue, element);
	if (!v.first)
		MIRA_THROW(XInvalidConfig, v.second);
	return *v.first;
}

bool hasElement(const Value& iValue, const std::string& element)
{
	std::pair<const Value*, std::string> v = accessElement(iValue, element);
	return v.first != NULL;
}

QueryValueResult getElementIfExists(const Value& iValue, const std::string& element)
{
	std::pair<const Value*, std::string> v = accessElement(iValue, element);

	QueryValueResult r;
	if (v.first) {
		r.exists = true;
		r.value = *v.first;
	} else {
		r.exists = false;
		r.error = v.second;
	}

	return r;
}

std::tuple<bool, size_t, size_t, std::string> trailing2DIndex(const std::string& element)
{
	// check if element is of form a.b...x[i1][i2], if so return head a.b...x and indices i1, i2

	const size_t npos = std::string::npos;

	size_t startAt = element.rfind('.')+1; // start behind last '.' in element
	                                       // or, if none found, at 0 (npos + 1 = 0)

	size_t bracket1 = element.find('[', startAt);
	if (bracket1 == npos)
		return std::make_tuple(false, 0, 0, std::string());

	size_t bracket2 = element.find("][", bracket1+1);
	if (bracket2 == npos)
		return std::make_tuple(false, 0, 0, std::string());

	size_t bracket3 = element.find(']', bracket2+1);
	if (bracket3 == npos)
		return std::make_tuple(false, 0, 0, std::string());

	if ((bracket2 < bracket1+2) || (bracket3 < bracket2+2))
		MIRA_THROW(XInvalidConfig, "json::getNumberElement(): Wrong syntax for 2D matrix element "
		                           "(use [idx1][idx2])");

	if (bracket3 != element.size()-1)
		MIRA_THROW(XInvalidConfig, "json::getNumberElement(): Wrong syntax for 2D matrix element "
		                           "(must end with [idx1][idx2])");

	size_t index1, index2;
	try {
		index1 = fromString<size_t>(element.substr(bracket1+1, bracket2-bracket1-1));
		index2 = fromString<size_t>(element.substr(bracket2+2, bracket3-bracket2-2));
	}
	catch(XIO& ex) {
		MIRA_RETHROW(ex, "json::getNumberElement(): Wrong syntax for 2D matrix element "
		                 "(use [idx1][idx2])");
	}

	return std::make_tuple(true, index1, index2, element.substr(0, bracket1));
}

std::pair<double, std::string> accessNumberElement(const Value& iValue, const std::string& element)
{
	bool accessMatrixElement;
	size_t idx1, idx2;
	std::string head;

	std::tie(accessMatrixElement, idx1, idx2, head) = trailing2DIndex(element);

	if (accessMatrixElement) {
		std::pair<const Value*, std::string> v = accessElement(iValue, head);
		if (!v.first)
			return std::pair<double, std::string>(0.0, v.second);

		const Value* value = v.first;
		if (value->type() != json_spirit::str_type)
			return std::pair<double, std::string>(
				0.0, "json::getNumberElement(): Data must be a string "
				     "(representing a matrix) if 2D index is specified");
		Eigen::MatrixXf m;
		std::stringstream ss(value->get_str());
		ss >> format(m);
		if ((int)idx1 >= m.rows() || (int)idx2 >= m.cols())
			return std::pair<double, std::string>(
				0.0, "json::getNumberElement(): Matrix element index is out of range");
		return std::pair<double, std::string>(m(idx1, idx2), "");
	}

	std::pair<const Value*, std::string> v = accessElement(iValue, element);
	if (!v.first)
		return std::pair<double, std::string>(0.0, v.second);

	const Value* value = v.first;

	// special treatment for bool
	if (value->type() == json_spirit::bool_type)
		return std::pair<double, std::string>(value->get_bool()?1.0:0.0, "");
	if (value->type() != json_spirit::int_type &&
		value->type() != json_spirit::real_type)
		return std::pair<double, std::string>(
			0.0, "json::getNumberElement(): Data must be of real, int or bool type");
	return std::pair<double, std::string>(value->get_value<double>(), "");
}

double getNumberElement(const Value& iValue, const std::string& element)
{
	std::pair<double, std::string> v = accessNumberElement(iValue, element);
	if (!v.second.empty())
		MIRA_THROW(XInvalidConfig, v.second);

	return v.first;
}

bool hasNumberElement(const Value& iValue, const std::string& element)
{
	std::pair<double, std::string> v = accessNumberElement(iValue, element);
	return v.second.empty();
}

QueryNumberResult getNumberElementIfExists(const Value& iValue, const std::string& element)
{
	std::pair<double, std::string> v = accessNumberElement(iValue, element);

	QueryNumberResult r;
	if (v.second.empty()) {
		r.exists = true;
		r.number = v.first;
	} else {
		r.exists = false;
		r.error = v.second;
	}

	return r;
}

///////////////////////////////////////////////////////////////////////////////

}} // namespace
