/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Colormaps.C
 *	implementation of the colormap classes.
 *
 * @author Erik Einhorn
 * @date   2011/01/25
 */

#include <factory/Factory.h>
#include <image/Colormap.h>
#include <math/Power.h>
#include <math/Saturate.h>

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_REGISTER(mira::Colormap, mira::Object)

MIRA_CLASS_REGISTER(mira::DiscreteColormap, mira::Colormap)
MIRA_CLASS_REGISTER(mira::ContinuousColormap, mira::Colormap)
MIRA_CLASS_REGISTER(mira::TabularColormap, mira::DiscreteColormap)

MIRA_CLASS_REGISTER(mira::GrayscaleColormap, mira::ContinuousColormap)
MIRA_CLASS_REGISTER(mira::HSVColormap, mira::ContinuousColormap)
MIRA_CLASS_REGISTER(mira::JetColormap, mira::ContinuousColormap)
MIRA_CLASS_REGISTER(mira::RedBlueColormap, mira::ContinuousColormap)
MIRA_CLASS_REGISTER(mira::Pm3dColormap, mira::ContinuousColormap)

MIRA_CLASS_REGISTER(mira::Complementary10Colormap, mira::TabularColormap)
MIRA_CLASS_REGISTER(mira::Complementary6Colormap, mira::TabularColormap)

///////////////////////////////////////////////////////////////////////////////

namespace mira {

///////////////////////////////////////////////////////////////////////////////

Color::RGB GradientColormapBase::getf(float f) const
{
	if(mColorKeys.empty())
		return Color::Black;

	// find first key that is after f
	auto after = mColorKeys.begin();
	while(after!=mColorKeys.end() && after->first<=f)
		++after;

	if(after==mColorKeys.begin())
		return mColorKeys.begin()->second;
	if(after==mColorKeys.end())
		return mColorKeys.rbegin()->second;

	auto before = after; before--;

	// interpolate between 'before' and 'after'
	float alpha = (f - before->first) / (after->first - before->first);
	return Color::RGB(
			(1.0f-alpha)*before->second.r + alpha*after->second.r,
			(1.0f-alpha)*before->second.g + alpha*after->second.g,
			(1.0f-alpha)*before->second.b + alpha*after->second.b);
}

void GradientColormapBase::setColorAt(float position, const Color::RGB& color)
{
	mColorKeys[position] = color;
}

void GradientColormapBase::clear()
{
	mColorKeys.clear();
}

///////////////////////////////////////////////////////////////////////////////

static float colorFormula(float x, int formula)
{
	const float p90  = 0.5f*pi<float>();
	const float p180 = 1.0f*pi<float>();
	const float p360 = 2.0f*pi<float>();
	const float p720 = 4.0f*pi<float>();

	switch(formula) {
	case 0: return 0.0f;
	case 1: return 0.5f;
	case 2: return 1.0f;
	case 3: return x;
	case 4: return pow2(x);
	case 5: return pow3(x);
	case 6: return pow<4>(x);
	case 7: return std::sqrt(x);
	case 8: return std::sqrt(std::sqrt(x));
	case 9: return std::sin(p90*x);
	case 10: return std::cos(p90*x);
	case 11: return std::abs(x-0.5f);
	case 12: return pow2(2.0f*x-1.0f);
	case 13: return std::sin(p180*x);
	case 14: return std::abs(std::cos(p180*x));
	case 15: return std::sin(p360*x);
	case 16: return std::cos(p360*x);
	case 17: return std::abs(std::sin(p360*x));
	case 18: return std::abs(std::cos(p360*x));
	case 19: return std::abs(std::sin(p720*x));
	case 20: return std::abs(std::cos(p720*x));
	case 21: return 3.0f*x;
	case 22: return 3.0f*x-1.0f;
	case 23: return 3.0f*x-2.0f;
	case 24: return std::abs(3.0f*x-1.0f);
	case 25: return std::abs(3.0f*x-2.0f);
	case 26: return (3.0f*x-1.0f)/2.0f;
	case 27: return (3.0f*x-2.0f)/2.0f;
	case 28: return std::abs((3.0f*x-1.0f)/2.0f);
	case 29: return std::abs((3.0f*x-2.0f)/2.0f);
	case 30: return x/0.32f-0.78125f;
	case 31: return 2.0f*x-0.84f;
	case 32: return 0.0f; // 4x;1;-2x+1.84;x/0.08-11.5???
	case 33: return std::abs(2.0f*x - 0.5f);
	case 34: return 2.0f*x;
	case 35: return 2.0f*x - 0.5f;
	case 36: return 2.0f*x - 1.0f;
	}
	return 0.0f;
}

Color::RGB FormulaeColormapBase::getf(float f) const
{
	return Color::RGB(
	            saturate(colorFormula(f,mRFormula),0.0f,1.0f),
	            saturate(colorFormula(f,mGFormula),0.0f,1.0f),
	            saturate(colorFormula(f,mBFormula),0.0f,1.0f));
}

void FormulaeColormapBase::setFormulae(int rformula, int gformula, int bformula)
{
	mRFormula = rformula;
	mGFormula = gformula;
	mBFormula = bformula;
}

///////////////////////////////////////////////////////////////////////////////

Pm3dColormap::Pm3dColormap(std::size_t size) : FormulaeColormapBase(size)
{
	setFormulae(7,5,15);
}

///////////////////////////////////////////////////////////////////////////////

const uint8 Complementary10Colormap::colormap[3*10] =
{
	0  , 0  , 0,
	0  , 0  , 255,
	255, 0  , 255,
	255, 0  , 0,
	255, 128, 0,
	255, 255, 0,
	192, 255, 0,
	0  , 255, 0,
	0  , 255, 255,
	255, 255, 255
};

///////////////////////////////////////////////////////////////////////////////

const uint8 Complementary6Colormap::colormap[3*6] =
{
	0  , 0  , 0,
	0  , 0  , 255,
	0  , 255, 0,
	255, 255, 0,
	255, 0  , 0,
	255, 255, 255
};

///////////////////////////////////////////////////////////////////////////////
}
