/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LibraryRegistry.C
 *    Implementation of LibraryRegister.h.
 *
 * @author Christof Schröter
 * @date   2020/10/04
 */

#include <factory/LibraryRegistry.h>

#include <factory/Class.h>
#include <error/Logging.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

std::ostream& operator<<(std::ostream& s, const LibraryVersion& version)
{
	s << version.mMajor << "." << version.mMinor << "." << version.mPatch;
	return s;
}

void LibraryRegistry::registerManifest( const std::string& name,
                                        const LibraryVersion& version )
{
	LibraryRegistry& registry = instance();
	boost::mutex::scoped_lock lock(registry.mMutex);
	LibraryInfo& lib = registry.mLibraries[name];
	if (lib.mHasManifest)
		MIRA_THROW( XFactoryLogical, "Manifest already registered for library: " << name);

	if ((lib.mIsLoaded) && (version != lib.mVersion))
		MIRA_THROW( XFactoryLogical,
		            "Registering manifest for library loaded before with different version: " <<
		            name <<
		            ", manifest version " << lib.mVersion <<
		            ", loaded library version " << version);

	lib.mVersion = version;
	lib.mHasManifest = true;
}

void LibraryRegistry::registerLibrary( const std::string& name,
                                       const LibraryVersion& version )
{
	LibraryRegistry& registry = instance();
	boost::mutex::scoped_lock lock(registry.mMutex);
	LibraryInfo& lib = registry.mLibraries[name];
	if (lib.mIsLoaded) {
		MIRA_LOG(ERROR) << "Library already loaded: " << name;
		return;
	}

	if ((lib.mHasManifest) && (version != lib.mVersion)) {
		MIRA_LOG(ERROR) << "Loading library with different version than manifest: " << name <<
		                   ", manifest version " << lib.mVersion <<
		                   ", loaded library version " << version;
		return;
	}

	lib.mVersion = version;
	lib.mIsLoaded = true;
}

//////////////////////////////////////////////////////////////////////////////

} // namespace
