/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file StampedDataQueue.h
 *    Queue for Stamped data.
 *
 * @author Erik Einhorn
 * @date   2010/09/07
 */

#ifndef _MIRA_STAMPEDDATAQUEUE_H_
#define _MIRA_STAMPEDDATAQUEUE_H_

#include <error/Exceptions.h>
#include <utils/Stamped.h>

#include <math/Eigen.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Implements a queue where @ref Stamped data can be added.
 * Afterwards the data can be obtained by specifying an
 * arbitrary timestamp. The queue will then search within
 * the stored data to return the data that corresponds to
 * the requested timestamp.
 *
 * In order to control the number of elements in the queue, a
 * maximum number of items can be specified (the capacity).
 * Additionally, a maximum storage time span can be specified.
 * This ensures that data is removed from the queue if it is older
 * than the timestamp of the latest entry plus the max. storage
 * time.
 *
 * @cond INTERNAL
 * The data is stored with decreasing timestamps, i.e. the first
 * item in the container is the latest entry, while the last item
 * is the oldest entry. The timestamps of the items are guaranteed to
 * be strictly monotonic decreasing (from latest to the oldest).
 *
 * Note: iterator++ moves into the PAST, while iterator-- moves
 *       into the future towards the present moment.
 * @endcond
 */
template <typename T >
class StampedDataQueue
{
public:

	/// The type of the underlying container
	typedef std::list<Stamped<T>, Eigen::aligned_allocator<T>> Container;

	/// some typedefs for STL compliance
	typedef Stamped<T> value_type;

	/// some typedefs for STL compliance
	typedef typename Container::const_iterator const_iterator;

	/// some typedefs for STL compliance
	typedef typename Container::size_type size_type;

public:

	/**
	 * Creates new StampedDataQueue with the specified maximum capacity
	 * (default 100) and maximum storage time (default 10 seconds).
	 */
	StampedDataQueue(size_type maxCapacity=100,
	                 Duration maxStorageTime = Duration::seconds(10) ) :
		mMaxCapacity(maxCapacity), mMaxStorageTime(maxStorageTime) {}

public:

	/**
	 * Sets the maximum storage time.
	 */
	void setMaxStorageTime(Duration maxStorageTime)
	{
		mMaxStorageTime = maxStorageTime;
		prune();
	}

	// STL conform operations:

	/**
	 * Returns the maximum number of items that can be stored in the queue.
	 * If more items are added, the oldest items will get lost.
	 */
	size_type capacity() const { return mMaxCapacity; }

	/**
	 * Request a change in capacity.
	 * Requests that the capacity of the queue for the elements of the vector
	 * container be at least enough to hold n elements.
	 *
	 * @note The capacity only can grow.
	 */
	void reserve(size_type n) {
		if(n>mMaxCapacity)
			mMaxCapacity=n;
	}


	/// Returns the number of stored elements
	size_type size() const { return mStorage.size(); }

	/// Returns true, if no element is stored
	bool empty() const { return mStorage.empty(); }


	/**
	 * Inserts new data to the queue.
	 * Returns true, if the data was inserted.
	 * Returns false, if the data is too old and was not inserted into the queue.
	 */
	bool insert(const Stamped<T>& data)
	{
		typename Container::iterator it = mStorage.begin();

		// check if new data is too old, if so, then skip it
		if(it!=mStorage.end() && it->timestamp > data.timestamp + mMaxStorageTime)
			return false;

		// find position to insert the data
		for(;it != mStorage.end(); ++it)
		{
			// if the timestamp of the item at "it" is older, then we need to insert
			// before "it"
			if (data.timestamp >= it->timestamp)
				break;
		}

		// guarantee STRICTLY monotonic decreasing and overwrite
		// an existing entry with the same timestamp, if any!
		if(it!=mStorage.end() && it->timestamp==data.timestamp)
			*it = data;
		else {
			// if data is older than all elements in the queue and queue is 
			// already full skip insertion
			if (it == mStorage.end() && mStorage.size() == mMaxCapacity)
				return false;
			// otherwise insert the data before "it"
			mStorage.insert(it, data);

			// prune the data (throw away oldest elements if necessary)
			prune();
		}

		return true;
	}

	/// clears the whole queue and removes all elements
	void clear() { mStorage.clear(); }

public:

	/**
	 * Returns the data from the queue whose time stamp is closest to the
	 * specified timestamp.
	 */
	const Stamped<T>& getData(const Time& timestamp) const
	{
		if (mStorage.empty())
			MIRA_THROW(XRuntime, "Trying to get data from empty StampedDataQueue");

		const_iterator older = findFirstOlderItem(timestamp);
		// older already is the first element -> there is no newer one
		if(older==mStorage.begin()) 
			return *older;

		const_iterator newer = older;
		--newer;

		// there is no older element return the newer one (oldest in queue)
		if(older==mStorage.end())
			return *newer;

		// return closest value
		if( (newer->timestamp - timestamp) <
			(timestamp - older->timestamp))
			return *newer;
		else
			return *older;
	}

	/**
	 * Returns two iterators that mark the begin and end of an interval of data.
	 * The interval is specified by the four parameters, where 'size' specifies
	 * the desired length of the interval. The 'before' and 'after' parameter
	 * specify the desired number of data items before (=older than) and after
	 * (=newer than) the specified 'timestamp'. If 'before'+'after' is lower
	 * than 'size', the method will add additional items before and after the
	 * given time stamp in order to fulfill the 'size' constraint, if possible.
	 *
	 * If there is a data item in the queue with exactly the requested timestamp,
	 * it counts on the after side.
	 *
	 * @note The second returned iterator marks the end of the requested
	 * interval, i.e. like with container ends its predecessor is the last valid
	 * member of that interval itself, fulfilling the before constraint
	 */
	std::pair<const_iterator,const_iterator> getInterval(const Time& timestamp,
	                                                     std::size_t size,
	                                                     std::size_t before,
	                                                     std::size_t after)
	{
		assert(size==0 || size >= before+after);

		const_iterator older = findFirstOlderItem(timestamp);

		// |----|-----|------------|------|-----|
		// ^              ^        ^            ^
		// begin        timestamp older         end

		// move backward in time until we reach the end of the buffer, 
		// or got the requested number of slots
		const_iterator last = older;
		for(std::size_t i=0; i<before && last!=mStorage.end(); ++i, ++last) ;

		const_iterator first = older;
		std::size_t i = 0;
		for(; i<after && first!=mStorage.begin(); ++i, --first) ;

		// now ensure the size constraint (if there's one)
		if(size!=0) {
			std::size_t left = size-std::distance(first,last);
			while(left>0 && first!=mStorage.begin()) {
				--first;
				--left;
			}
			while(left>0 && last!=mStorage.end()) {
				++last;
				--left;
			}

			if(left>0)
				MIRA_THROW(XRuntime,
				           "Not enough items in StampedDataQueue to obtain "
				           "the requested interval: " << left <<
				           " items missing.")
		}

		return std::make_pair(first, last);
	}

private:

	/**
	 * Removes entries that are too old, or if there are too many entries, then
	 * it removes the oldest ones.
	 */
	void prune()
	{
		// the oldest "allowed" timestamp
		Time oldestTimestamp = mStorage.begin()->timestamp - mMaxStorageTime;

		// remove the elements at the end that are too old or if there are too many
		while(!mStorage.empty() && 
		      (mStorage.rbegin()->timestamp < oldestTimestamp ||
		       mStorage.size() > mMaxCapacity) )
			mStorage.pop_back();
	}


	// find the first iterator that is older than our searched timestamp
	typename Container::const_iterator findFirstOlderItem(const Time& timestamp) const
	{
		// move from newest to oldest slot
		for(typename Container::const_iterator it=mStorage.begin();
				                               it!=mStorage.end(); ++it)
			if(it->timestamp < timestamp)
				return it;

		// if queue is empty or the oldest queue element is already newer than
		// the requested timestamp, we end up here
		return mStorage.end();
	}

private:

	size_type  mMaxCapacity;
	Duration mMaxStorageTime;

	Container mStorage;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
