/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file EnumToFlags.h
 *    Macros for generating logical operators for using enum values as flags.
 *
 * @author Erik Einhorn
 * @date   2010/06/28
 */

#ifndef _MIRA_ENUMTOFLAGS_H_
#define _MIRA_ENUMTOFLAGS_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Macro that can be used with enums that contain flags. The macro generates
 * code for the logical operators &,|,^,~ to allow the enum values to be
 * used as normal flags which would lead to compiler errors if these operators
 * were not defined.
 *
 * Example:
 * \code
 *     enum MyFlags
 *     {
 *         flag1 = 0x0001,
 *         flag2 = 0x0002,
 *         flag3 = 0x0004,
 *         flag4 = 0x0008
 *     }
 *
 *     MIRA_ENUM_TO_FLAGS(MyFlags) // generate logical operators for enum values
 *
 *     ...
 *
 *     MyFlags combined = flag1 | flag2 & flag4;
 *     if(combined & flag3) {
 *         ...
 *     }
 * \endcode
 */
#define MIRA_ENUM_TO_FLAGS(EnumType)                                 \
	inline EnumType                                                 \
	operator&(EnumType a, EnumType b)                               \
	{ return EnumType(static_cast<int>(a) & static_cast<int>(b)); } \
	                                                                \
	inline EnumType                                                 \
	operator|(EnumType a, EnumType b)                               \
	{ return EnumType(static_cast<int>(a) | static_cast<int>(b)); } \
	                                                                \
	inline EnumType                                                 \
	operator^(EnumType a, EnumType b)                               \
	{ return EnumType(static_cast<int>(a) ^ static_cast<int>(b)); } \
	                                                                \
	inline EnumType&                                                \
	operator|=(EnumType& a, EnumType b)                             \
	{ return a = a | b; }                                           \
	                                                                \
	inline EnumType&                                                \
	operator&=(EnumType& a, EnumType b)                             \
	{ return a = a & b; }                                           \
	                                                                \
	inline EnumType&                                                \
	operator^=(EnumType& a, EnumType b)                             \
	{ return a = a ^ b; }                                           \
	                                                                \
	inline EnumType                                                 \
	operator~(EnumType a)                                           \
	{ return EnumType(~static_cast<int>(a)); }



/**
 * Macro that can be used with enums that contain flags. The macro generates
 * code for the logical operators &,|,^,~ to allow the enum values to be
 * used as normal flags which would lead to compiler errors if these operators
 * were not defined.
 * This macro works just as the MIRA_ENUM_TO_FLAGS macro but can be used
 * for enums that are defined in a class.
 *
 * Example:
 * \code
 *
 *     class MyClass
 *     {
 *     public:
 *         enum MyFlags
 *         {
 *             flag1 = 0x0001,
 *             flag2 = 0x0002,
 *             flag3 = 0x0004,
 *             flag4 = 0x0008
 *         }
 *         MIRA_ENUM_TO_FLAGS_INCLASS(MyFlags) // generate logical operators for enum values
 *     };
 *
 *     ...
 *
 *     MyClass::MyFlags combined = flag1 | flag2 & flag 4;
 *     if(combined & flag3) {
 *         ...
 *     }
 * \endcode
 */
#define MIRA_ENUM_TO_FLAGS_INCLASS(EnumType)                         \
	friend inline EnumType                                          \
	operator&(EnumType a, EnumType b)                               \
	{ return EnumType(static_cast<int>(a) & static_cast<int>(b)); } \
	                                                                \
	friend inline EnumType                                          \
	operator|(EnumType a, EnumType b)                               \
	{ return EnumType(static_cast<int>(a) | static_cast<int>(b)); } \
	                                                                \
	friend inline EnumType                                          \
	operator^(EnumType a, EnumType b)                               \
	{ return EnumType(static_cast<int>(a) ^ static_cast<int>(b)); } \
	                                                                \
	friend inline EnumType&                                         \
	operator|=(EnumType& a, EnumType b)                             \
	{ return a = a | b; }                                           \
	                                                                \
	friend inline EnumType&                                         \
	operator&=(EnumType& a, EnumType b)                             \
	{ return a = a & b; }                                           \
	                                                                \
	friend inline EnumType&                                         \
	operator^=(EnumType& a, EnumType b)                             \
	{ return a = a ^ b; }                                           \
	                                                                \
	friend inline EnumType                                          \
	operator~(EnumType a)                                           \
	{ return EnumType(~static_cast<int>(a)); }                      \

///////////////////////////////////////////////////////////////////////////////

}

#endif
