/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Buffer.h
 *    Generic buffer class that can be used as a replacement for std::vector.
 *
 * @author Erik Einhorn
 * @date   2010/11/19
 */

#ifndef _MIRA_BUFFER_H_
#define _MIRA_BUFFER_H_

#include <memory>
#include <stdexcept>
#include <vector>

#include <platform/Types.h>

#ifndef Q_MOC_RUN
#include <boost/type_traits.hpp>
#endif

#include <serialization/ReflectorInterface.h>
#include <serialization/Array.h>
#include <serialization/IsTransparentSerializable.h>
#include <serialization/ReflectorMacros.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Generic buffer class that can be used as a replacement for std::vector
 * whenever copying and reallocation of internal array buffer needs to be
 * avoided.
 * The Buffer has the following advantages over the std::vector:
 *  - Buffers can store pointers to already allocated memory
 *    without copying and owning it.
 *  - Memory/Size of Buffers can be reserved and resized later when the real
 *    size is known without allocating, freeing or copying memory when the
 *    new size is smaller than the reserved size.
 *    This has the advantage of only allocating memory once when reserving a
 *    buffer of the maximum expected size and using resize every time data is
 *    copied into the buffer later.
 */
template <typename T, typename Alloc = std::allocator<T> >
class Buffer
{
	// the is_pod check is too hard here, since user defined constructors
	// are allowed, as long there is a trivial default constructor,
	// copy constructor and trivial destructor
	static_assert(boost::has_trivial_destructor<T>::value,
			"Can be used with types that have trivial destructor only");
#ifndef MIRA_WINDOWS
	// VC++2010 does not yet support the "Explicitly-defaulted and deleted
	// special member functions"-syntax, therefore we must skip these checks
	static_assert(boost::has_trivial_default_constructor<T>::value,
			"Can be used with types that have a trivial default constructor only");
	static_assert(boost::has_trivial_copy_constructor<T>::value,
			"Can be used with types that have a trivial copy constructor only");
#endif

	// TODO: The standard library has renamed all of the above has_trivial_XYZ to
	//       is_trivially_XYZable WITHOUT providing the old versions for
	//       backward compatibility. Therefore, the following code works on
	//       GCC >= 4.8 but fails on previous GCCs versions and MSVC. Therefore,
	//       we prefer to use the boost variants above, which seem to be more
	//       stable than the standard
	//static_assert(std::is_trivially_destructible<T>::value,
	//		"Can be used with types that have trivial destructor only");
	//static_assert(std:::is_trivially_default_constructible<T>::value,
	//		"Can be used with types that have a trivial default constructor only");
	//static_assert(std::is_trivially_copy_constructible<T>::value,
	//		"Can be used with types that have a trivial copy constructor only");


public:

	// some STL conform typedefs
	typedef T                               value_type;
	typedef typename Alloc::pointer         pointer;
	typedef typename Alloc::const_pointer   const_pointer;
	typedef typename Alloc::reference       reference;
	typedef typename Alloc::const_reference const_reference;
	typedef T*                              iterator;
	typedef const T*                        const_iterator;
	typedef std::size_t                     size_type;
	typedef ptrdiff_t                       difference_type;
	typedef Alloc                           allocator_type;

private:

	void init() {
		mBuffer = NULL;
		mReserved = mSize = 0;
		mBufferCreated = false;
		mTakenVector = NULL;
	}

public:

	/// Default constructor constructing an empty buffer
	Buffer() {
		init();
	}

	/*
	 * Create a buffer and allocate memory of size size
	 */
	Buffer(size_type size) {
		init();
		resize(size);
	}

	/**
	 * Use already allocated memory in carray with size size.
	 * Buffer does not take ownership of carray.
	 * No data is copied unless calling reserve or resize with size > size of carray.
	 */
	Buffer(T* carray, size_type size) {
		mBuffer = carray;
		mReserved = mSize = size;
		mBufferCreated = false;
		mTakenVector = NULL;
	}

	/**
	 * Constructs a buffer that TAKES OVER all data from the specified vector.
	 * The content of the vector will be destroyed and will be completely moved
	 * to the created buffer without any necessary copying.
	 * Please note, that the vector is passed as rvalue reference. In order
	 * to move the content of a vector into the newly created buffer, you
	 * can construct the buffer as follows:
	 * \code
	 * std::vector<int> myvec;
	 * ... fill myvec ...
	 *
	 * // construct a new Buffer and take over all data of myvec, by "moving"
	 * // its content into the buffer:
	 * Buffer<int> mybuf(std::move(myvec));
	 *
	 * // myvec will now be empty, and its whole content is moved into the
	 * // buffer mybuf.
	 * \endcode
	 */
	Buffer(std::vector<T>&& other)
	{
		mTakenVector = new std::vector<T>;
		mTakenVector->swap(other);

		mBuffer = &(*mTakenVector)[0];
		mReserved = mTakenVector->capacity();
		mSize = mTakenVector->size();
		mBufferCreated = false;
	}

	/// Destructor
	~Buffer() {
		setBuffer(NULL, 0);
	}

public:

	/// Copy constructor
	Buffer(const Buffer& other) {
		init();
		copy(other);
	}

	/// Assignment
	Buffer& operator=(const Buffer& other) {
		copy(other);
		return *this;
	}

	/// Copies a buffer into this
	void copy(const Buffer& other) {
		resize(other.size());
		memcpy(mBuffer, other.data(), other.size()*sizeof(T));
	}

public:
	// Move semantic

	/// Move constructor
	Buffer(Buffer&& other) noexcept {
		init();
		swap(other);
	}

	/// Move assignment
	Buffer& operator=(Buffer&& other) noexcept {
		swap(other);
		return *this;
	}

	/// Swaps the content of this buffer with the other buffer
	void swap(Buffer& other) {
		std::swap(mBuffer, other.mBuffer);
		std::swap(mSize, other.mSize);
		std::swap(mReserved, other.mReserved);
		std::swap(mBufferCreated, other.mBufferCreated);
		std::swap(mTakenVector, other.mTakenVector);
	}

public:

	/// Checks for equality with other buffer
	bool operator==(const Buffer& other) const {
		if(size()!=other.size())
			return false;
		assert(sizeInBytes()==other.sizeInBytes());
		return memcmp(data(), other.data(), sizeInBytes()) == 0;
	}

	/// Checks for inequality with other buffer
	bool operator!=(const Buffer& other) const {
		return !operator==(other);
	}

public:

	/**
	 * Returns an iterator referring to the first element
	 * in the vector container.
	 */
	iterator begin() { return mBuffer; }

	/**
	 * Returns an iterator referring to the first element
	 * in the vector container.
	 */
	const_iterator begin() const { return mBuffer; }

	/**
	 * Returns an iterator referring to the past-the-end element
	 * in the vector container.
	 */
	iterator end() { return mBuffer+mSize; } // also works if empty

	/**
	 * Returns an iterator referring to the past-the-end element
	 * in the vector container.
	 */
	const_iterator end() const { return mBuffer+mSize; }

public:
	/**
	 * Returns the maximum size of the buffer
	 */
	size_type max_size () const { return mAllocator.max_size(); }

	/**
	 * Returns the used size of the buffer set by resize()
	 */
	size_type size() const { return mSize; }

	/**
	 * Returns the used size in bytes
	 */
	size_type sizeInBytes() const { return mSize * sizeof(T); }

	/**
	 * Checks if the buffer is empty (used size == 0).
	 */
	bool empty () const { return mSize==0; }

	/**
	 * Returns the reserved size/capacity of the buffer 
	 * (Its real size) set by reserve()
	 */
	size_type capacity () const { return mReserved; }

	/**
	 * Returns the allocator
	 */
	allocator_type get_allocator() const { return mAllocator; }

public:

	/**
	 * Allocates new memory if reserved size < new reserved size.
	 * Otherwise allocated memory stays the same.
	 * Reserved memory is the smallest power of two >= new reserved size. 
	 */
	void reserve(size_type reserve)
	{
		if(reserve > mReserved)
		{
			std::size_t newSize = mReserved;
			if(newSize==0)
				newSize=1;
			while(reserve > newSize)
				newSize *= 2;

			T* newBuffer = mAllocator.allocate(newSize);
			if(mBuffer!=NULL)
				memcpy(newBuffer, mBuffer, mSize*sizeof(T));
			setBuffer(newBuffer, newSize);
			mBufferCreated = true;
		}
	}

public:

	/**
	 * Resizes the buffer.
	 * Sets only size to given size if buffer reserved size is >= new size.
	 * Allocates more memory and copies existing data if reserved size < new size
	 * If size is increased, new fields are uninitialized and need to be
	 * filled e.g. by copying elements into them.
	 */
	void resize(size_type size)
	{
		reserve(size); // make sure buffer is big enough
		mSize = size;
	}

public:

	/**
	 * Adds a new element at the end of the vector, after its current last
	 * element. The content of this new element is initialized to a copy of x.
	 */
	void push_back(const T& x) {
		resize(mSize+1);
		mBuffer[mSize-1] = x;
	}

	/**
	 * Adds new elements to the end of the vector, after its current last element.
	 * The content of these new elements is initialized to a copy of data.
	 */
	void push_back(T* data, size_type size) {
		push_back(Buffer<T>(data, size));
	}

	/**
	 * Adds new elements to the end of the vector, after its current last element.
	 * The content of these new elements is initialized to a copy of data.
	 */
	void push_back(const Buffer<T>& data) {
		size_type oldSize = mSize;
		resize(mSize + data.size());
		memcpy(mBuffer+oldSize*sizeof(T), data.data(), data.sizeInBytes());
	}

	/**
	 * Removes the last element in the vector, effectively reducing the vector
	 * size by one. Does not free any memory but an iterator to the removed element
	 * will become invalid.
	 */
	void pop_back() {
		pop_back(1);
	}

	/**
	 * Removes the last elements in the vector, effectively reducing the vector
	 * size by elements. Does not free any memory but iterators to removed elements
	 * will become invalid.
	 */
	void pop_back(size_type elements) {
		if (elements >= mSize)
		{
			this->clear();
			return;
		}
		resize(mSize-elements);
	}

	/**
	 * Removes the first element in the vector by copying all remaining data to
	 * the front invalidating all iterators and references to the buffer.
	 */
	void pop_front() {
		pop_front(1);
	}

	/**
	 * Removes the first elements in the vector by copying all remaining data to
	 * the front invalidating all iterators and references to the buffer.
	 */
	void pop_front(size_type elements) {
		if (elements >= mSize)
		{
			clear();
			return;
		}
		// Move the memory
		if (elements > 0)
		{
			memmove(mBuffer, mBuffer+elements*sizeof(T), (mSize-elements)*sizeof(T));
			resize(mSize-elements);
		}
	}

	/**
	 * All the elements of the vector are dropped.
	 */
	void clear() {
		mSize = 0;
	}

public:

	/**
	 * Returns a reference to the element at position n in the vector container.
	 */
	reference operator[] ( size_type n ) {
		assert(n<mSize);
		return mBuffer[n];
	}

	/**
	 * Returns a reference to the element at position n in the vector container.
	 */
	const_reference operator[] ( size_type n ) const {
		assert(n<mSize);
		return mBuffer[n];
	}

	/**
	 * The difference between this member function and member operator function
	 * operator[] is that vector::at signals if the requested position is out of
	 * range by throwing an out_of_range exception.
	 */
	reference at(size_type n) {
		rangeCheck(n);
		return mBuffer[n];
	}

	/**
	 * The difference between this member function and member operator function
	 * operator[] is that vector::at signals if the requested position is out of
	 * range by throwing an out_of_range exception.
	 */
	const_reference at(size_type n) const {
		rangeCheck(n);
		return mBuffer[n];
	}

	/**
	 *  Returns a reference to the first element in the vector container.
	 */
	reference front() {
		assert(mSize>0);
		return mBuffer[0];
	}

	/**
	 *  Returns a reference to the first element in the vector container.
	 */
	const_reference front() const {
		assert(mSize>0);
		return mBuffer[0];
	}

	/**
	 *  Returns a reference to the last element in the vector container.
	 */
	reference back() {
		assert(mSize>0);
		return mBuffer[size()-1];
	}

	/**
	 * Returns a reference to the last element in the vector container.
	 */
	const_reference back() const {
		assert(mSize>0);
		return mBuffer[size()-1];
	}

	/**
	 *  Returns a pointer to the underlying data
	 */
	pointer data() { return mBuffer; }

	/**
	 * Returns a const pointer to the underlying data
	 */
	const_pointer data() const { return mBuffer; }

protected:

	///@cond INTERNAL
	void rangeCheck(size_type n) const
	{
		if (n >= size())
			throw std::out_of_range("Buffer::range_check");
	}
	///@endcond

private:

	///@cond INTERNAL
	void setBuffer(T* newBuffer, size_type size)
	{
		// delete old buffer, if any
		if(mBufferCreated)
			mAllocator.deallocate(mBuffer,mReserved);

		// data might have come from a taken over vector, which now is no longer needed
		delete mTakenVector;
		mTakenVector = NULL;

		mBuffer = newBuffer;
		mReserved = size;
	}
	///@endcond

protected:

	T* mBuffer;			///< Pointer to the data buffer
	size_type mSize;	///< The used elements of the buffer
	size_type mReserved;///< The real size of the buffer
	bool mBufferCreated;///< Was the buffer created (is it owned) by us

	Alloc mAllocator;	///< The allocator used to allocate new memory

	std::vector<T>* mTakenVector; ///< Used to take over the data from a vector without copying
};

///////////////////////////////////////////////////////////////////////////////

/// Specialization of the non-intrusive reflect for Buffer
template<typename Reflector, typename T, typename Allocator>
void reflectRead(Reflector& r, Buffer<T, Allocator>& c)
{
	// store the size first
	uint32 count = c.size();
	{
		typedef Buffer<T, Allocator> Buffer;
		MIRA_REFLECT_CALL(Reflector, r, "Buffer ReflectCollectionCount",
			              (serialization::ReflectCollectionCount<Reflector,Buffer>::reflect(r, count)));
	}
	// store the elements as array
	serialization::PlainArray<T> array(c.data(), c.size());
	r.delegate(array);
}

/// Specialization of the non-intrusive reflect for Buffer
template<typename Reflector, typename T, typename Allocator>
void reflectWrite(Reflector& r, Buffer<T, Allocator>& c)
{
	// restore the size first
	uint32 count;
	{
		typedef Buffer<T, Allocator> Buffer;
		MIRA_REFLECT_CALL(Reflector, r, "Buffer ReflectCollectionCount",
			              (serialization::ReflectCollectionCount<Reflector,Buffer>::reflect(r, count)));
	}
	// reserve the space and create the elements
	c.resize(count);
	// restore the elements as array
	serialization::PlainArray<T> array(c.data(), c.size());
	r.delegate(array);
}

/// non-intrusive reflect for Buffer
template<typename Reflector, typename T, typename Allocator>
void reflect(Reflector& r, Buffer<T, Allocator>& c)
{
	splitReflect(r, c);
}

template<typename T, typename Allocator>
class IsObjectTrackable<Buffer<T, Allocator>> : public std::false_type {};

template<typename T, typename Allocator>
class IsCollection<Buffer<T, Allocator>> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

}

#endif
