/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VoidCast.h
 *    Provides safe casts for casting from a pointer to void* and vice versa
 *    while taking care of polymorphism and multiple inheritance.
 *
 * @author Erik Einhorn
 * @date   2011/06/21
 */

#ifndef _MIRA_VOIDCAST_H_
#define _MIRA_VOIDCAST_H_

#include <assert.h>
#include <type_traits>

#ifndef Q_MOC_RUN
#include <boost/mpl/eval_if.hpp>
#include <boost/mpl/identity.hpp>
#endif

#include <factory/Object.h>

namespace mira { namespace serialization {

///////////////////////////////////////////////////////////////////////////////
//@cond INTERNAL
// for internal use by void casts.

struct VoidUpcastNormal {
	template <typename T>
	static void* invoke(T* pointer) {
		typedef typename std::remove_const<T>::type Tc;
		Tc* p = const_cast<Tc*>(pointer);
		return static_cast<void*>(p);
	}
};

struct VoidDowncastNormal {
	template <typename T>
	static T* invoke(void* pointer) {
		return static_cast<T*>(pointer);
	}
};

// For polymorphic types we cast via the base mira::Object type. This ensures
// safety even for multiple inheritance, since the address of the mira::Object
// part is "fixed" for all classes along the type hierarchy.

struct VoidUpcastPolymorphic {
	template <typename T>
	static void* invoke(T* pointer) {
		typedef typename std::remove_const<T>::type Tc;
		Tc* p = const_cast<Tc*>(pointer);
		mira::Object* obj = p;
		return static_cast<void*>(obj);
	}
};

struct VoidDowncastPolymorphic {
	template <typename T>
	static T* invoke(void* pointer) {
		mira::Object* obj = static_cast<mira::Object*>(pointer);
		T* p = dynamic_cast<T*>(obj);
		assert(p!=NULL); // this should never happen, since all polymorphic types
						 // MUST be derived from mira::Object
		return p;
	}
};
///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * Safe cast for casting from a pointer upwards to void*
 * while taking care of polymorphism and multiple inheritance.
 * To cast the void pointer upwards to the derived type use void_downcast.
 */
template <typename T>
void* void_upcast(T* pointer)
{
	typedef typename
	boost::mpl::eval_if<std::is_base_of<mira::Object, T>,
		boost::mpl::identity< VoidUpcastPolymorphic >,
	//else
		boost::mpl::identity< VoidUpcastNormal >
	>::type type;

	return type::template invoke<T>(pointer);
}

/**
 * Safe cast for casting from a void pointer to a derived pointer T*
 * while taking care of polymorphism and multiple inheritance.
 * The void pointer must have been casted down before using void_upcast.
 */
template <typename T>
T* void_downcast(void* pointer)
{
	typedef typename
	boost::mpl::eval_if<std::is_base_of<mira::Object, T>,
		boost::mpl::identity< VoidDowncastPolymorphic >,
	//else
		boost::mpl::identity< VoidDowncastNormal >
	>::type type;
	return type::template invoke<T>(pointer);
}

///////////////////////////////////////////////////////////////////////////////

} } // namespace

#endif
