/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolymorphicPointerReflector.h
 *    For internal use by RecursiveMemberReflector.h!
 *
 * @author Erik Einhorn
 * @date   2010/07/07
 */

#ifndef _MIRA_POLYMORPHICPOINTERREFLECTOR_H_
#define _MIRA_POLYMORPHICPOINTERREFLECTOR_H_

#include <map>

#include <utils/Foreach.h>
#include <serialization/VoidCast.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace serialization {

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for all derived PolymorphicPointerReflector classes for
 * different types.
 */
template <typename Derived>
struct PolymorphicPointerReflectorBase
{
	virtual ~PolymorphicPointerReflectorBase() {}

	/**
	 * Invokes the serializer on the pointer.
	 * This method is implemented in the derived PolymorphPointerReflector classes
	 * that are specialized for the registered polymorphic types.
	 */
	virtual void invoke(Derived& serializer, void* pointer) = 0;
};

/**
 * Template class that is specialized for all polymorphic types.
 * The specialization takes place when each polymorphic type is registered in
 * the PolymorphPointerReflectorRegistrar via the Serializer::registerClass()
 * and Deserializer::registerClass() method, usually triggered by the
 * invocation of the MIRA_CLASS_SERIALIZATION macro.
 * Hence there is a special PolymorphPointerReflector for each type. This
 * allows to "store" the type of the polymorphic pointer using a separate
 * instance of the specialized PolymorphPointerReflector.
 * The invoke method of the specialized PolymorphPointerReflector then
 * can cast the void* pointer safely to its real pointer type, stored in
 * the PolymorphPointerReflector object.
 */
template <typename T, typename Derived>
struct PolymorphicPointerReflector :
			public PolymorphicPointerReflectorBase<Derived>
{
	virtual void invoke(Derived& reflector, void* pointer)
	{
		// and serialize content recursively
		reflector.invokePointerObject(*serialization::void_downcast<T>(pointer));
	}
};

/**
 * Stores a special PolymorphPointerReflector object for each registered
 * polymorphic type. There is one PolymorphPointerReflectorRegistrar for each
 * existing Serializer/Deserializer class. And for each of these
 * PolymorphPointerReflectorRegistrar all polymorphic types need to be
 * registered.
 * Hence we need to create an overall number of M*N PolymorphPointerReflector
 * objects, where M is the number of Serializers/Deserializers and N the number
 * of polymorphic types. However, M usually is small (6-7) and there is no
 * other way to achieve the type-safety we need here. Boost::serialization and
 * s11n does it the same way.
 */
template <typename Derived>
class PolymorphicPointerReflectorRegistrar :
			public LazySingleton<PolymorphicPointerReflectorRegistrar<Derived> >
{
public:

	/// shared pointer to the registered PolymorphPointerReflector object
	typedef PolymorphicPointerReflectorBase<Derived>* ReflectorPtr;

public:

	~PolymorphicPointerReflectorRegistrar()	{

		if(mReflectors.size()>0) {
			// If we reach here, some classes were not unregistered before this
			// singleton is destroyed. We could life with that since it is
			// not a growing memory leak and the memory will be released upon
			// the termination of the process. However, to make tools like
			// valgrind happy, we release the memory, but we do NOT call the
			// proper destructor (therefore the char* cast). Calling the
			// destructor can be dangerous especially on Windows, since some
			// necessary dynamic libraries were already removed from memory
			// for some reason. This is somewhat hackish, but no problem at all
			// since the destructors do not do anything.
			// Maybe we find a better solution some day, but this has very low
			// priority.
			foreach(typename ReflectorMap::value_type& v, mReflectors)
				delete (char*)v.second;
		}
		mReflectors.clear();
	}

public:

	/**
	 * Registers a new polymorphic type for the Serializer/Desieralizer of
	 * the type "Derived".
	 */
	template<typename Class>
	void registerClass()
	{
		int typeId = Class::CLASS().getTypeId();
		if(mReflectors.count(typeId)!=0)
			return; // we already have registered that class type

		// add a new pointer serializer
		mReflectors.insert(std::make_pair(typeId,
		                                  new PolymorphicPointerReflector<Class, Derived>) );
	}

	/**
	 * Unregisters a polymorphic type of the Serializer/Desieralizer of the
	 * type "Derived".
	 */
	template<typename Class>
	void unregisterClass()
	{
		int typeId = Class::CLASS().getTypeId();

		auto it = mReflectors.find(typeId);
		if(it==mReflectors.end())
			return; // we have not registered that class type,
		            // or already unregistered it

		// delete the reflector
		delete it->second;

		// and remove it from our map
		mReflectors.erase(it);
	}

	/// Returns the registered PolymorphPointerReflector for the specified type.
	ReflectorPtr getPolymorphicPointerReflector(int typeId)
	{
		typename ReflectorMap::iterator i = mReflectors.find(typeId);
		if(mReflectors.find(typeId)==mReflectors.end())
			return ReflectorPtr(); // NULL
		return i->second;
	}

private:

	/// map with shared pointers to the registered PolymorphPointerReflector objects
	typedef std::map<int, ReflectorPtr > ReflectorMap;

	/// maps from typeids to pointer serializer objects for registered types
	ReflectorMap mReflectors;

};

///////////////////////////////////////////////////////////////////////////////

} // namespace
///@endcond

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
