/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Array.h
 *    Wrapper class for reflecting arrays.
 *
 * @author Erik Einhorn
 * @date   2010/09/28
 */

#ifndef _MIRA_ARRAY_H_
#define _MIRA_ARRAY_H_

#include <serialization/ReflectorInterface.h>
#include <serialization/SplitReflect.h>
#include <serialization/ReflectCollection.h>

#include <serialization/IsCollection.h>
#include <serialization/IsTransparentSerializable.h>
#include <serialization/IsObjectTrackable.h>

#include <utils/Foreach.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace serialization {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Class that is used by RecursiveMemberReflector to wrap around
 * arrays that are finally stored as collections item by item.
 *
 * This class acts as base implementation and is used by Array and other structures
 * behaving like arrays (std::vector).
 * In contrast to this class, the Array template class specifies the
 * IsCollection trait.
 */
template<typename T>
struct PlainArray
{
	typedef T value_type;

	PlainArray(T* iData, uint32 iSize) : data(iData), size(iSize) {}

	PlainArray(const PlainArray& other) : data(other.data), size(other.size) {}

	PlainArray operator=(const PlainArray& other) {
		if(size!=other.size)
			MIRA_THROW(XIO, "Cannot assign array with different size");
		// copy each item
		for(uint32 i=0; i<size; ++i)
			data[i]=other.data[i];
		return *this;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		typedef PlainArray<T> PlainArray;
		MIRA_REFLECT_CALL(Reflector, r, "PlainArray ReflectCollectionItems",
		                  (ReflectCollectionItems<Reflector, PlainArray>::reflect(r, *this)))
	}

	uint32 getSizeInBytes() const {
		return size * sizeof(value_type);
	}

	bool empty() const { return size==0; }

	T* begin() { return data; }
	T* end() { return data+size; }

	T* data;
	const uint32 size;
};

/**
 * Class that is used by RecursiveMemberReflector to wrap around (const-size!)
 * arrays that are finally stored as collections item by item.
 *
 * Uses PlainArray to wrap around an array. In contrast to PlainArray this class
 * explicitly reflects the number of array items.
 */
template<typename T>
struct Array
{
	Array(T* iData, uint32 iSize) : array(iData, iSize) {}

	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		uint32 count = array.size;
		typedef Array<T> Array;
		MIRA_REFLECT_CALL(Reflector, r, "Array ReflectCollectionCount",
		                  (ReflectCollectionCount<Reflector,Array>::reflect(r, count)));

		// delegate to array
		r.delegate(array);
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		uint32 count;
		typedef Array<T> Array;
		MIRA_REFLECT_CALL(Reflector, r, "Array ReflectCollectionCount",
		                  (ReflectCollectionCount<Reflector,Array>::reflect(r, count)));

		if(array.size!=count)
			MIRA_THROW(XIO, "Cannot reflect the collection with " << count <<
			                " items into an array of size " << array.size);

		// delegate to array
		r.delegate(array);
	}

	MIRA_SPLIT_REFLECT_MEMBER

	bool empty() const { return array.empty(); }

	T* begin() { return array.begin(); }
	T* end() { return array.end(); }

	PlainArray<T> array;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

// specify the internal marker traits for PlainArray and Array:
template <typename T>
class IsObjectTrackable<serialization::PlainArray<T>> : public std::false_type {};

template <typename T, typename SerializerTag>
class IsTransparentSerializable<serialization::PlainArray<T>,SerializerTag> : public std::true_type {};

template <typename T>
class IsObjectTrackable<serialization::Array<T>> : public std::false_type {};

template <typename T>
class IsCollection<serialization::Array<T>> : public std::true_type {};

///@endcond

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
